// Color Picker by kaelad02
// License: MIT
// Documentation: https://github.com/kaelad02/adv-reminder/blob/54207ec1ef0500439e57521f15956c07e4c02af4/src/settings.js#L91-L104

function colorPicker(settingId, html, settingValue) {
	const colorPickerElement = document.createElement("input");
	colorPickerElement.setAttribute("type", "color");
	colorPickerElement.setAttribute("data-edit", settingId);
	colorPickerElement.value = settingValue;

	// Add color picker
	const stringInputElement = html[0].querySelector(`input[name="${settingId}"]`);
	stringInputElement.classList.add("color");
	stringInputElement.after(colorPickerElement);
}

// injectConfig library by @theripper93
// License: MIT
// Documentation: https://github.com/theripper93/injectConfig

var injectConfig = {
	inject: function injectConfig(app, html, data, object) {
		this._generateTabStruct(app, html, data, object);
		const tabSize = data.tab?.width ?? 100;
		object = object || app.object;
		const moduleId = data.moduleId;
		let injectPoint;
		if (typeof data.inject === "string") {
			injectPoint = html.find(data.inject).first().closest(".form-group");
		} else {
			injectPoint = data.inject;
		}
		injectPoint = injectPoint ? $(injectPoint) : data.tab ? html.find(".tab[data-group='main']").last() : html.find(".form-group").last();
		let injectHtml = "";
		for (const [k, v] of Object.entries(data)) {
			if (k === "moduleId" || k === "inject" || k === "tab") continue;
			const elemData = data[k];
			const flag = `flags.${moduleId}.${k || ""}`;
			const flagValue = object?.getFlag(moduleId, k) ?? elemData.default ?? getDefaultFlag(k);
			const notes = v.notes ? `<p class="notes">${v.notes}</p>` : "";
			v.label = v.units ? `${v.label}<span class="units"> (${v.units})</span>` : v.label;
			switch (elemData.type) {
				case "text":
					injectHtml += `<div class="form-group">
                        <label for="${k}">${v.label || ""}</label>
                            <input type="text" name="${flag}" value="${flagValue}" placeholder="${v.placeholder || ""}">${notes}
                    </div>`;
					break;
				case "number":
					injectHtml += `<div class="form-group">
                        <label for="${k}">${v.label || ""}</label>
                            <input type="number" name="${flag}" min="${v.min}" max="${v.max}" step="${v.step ?? 1}" value="${flagValue}" placeholder="${
	v.placeholder || ""
}">${notes}
                    </div>`;
					break;
				case "checkbox":
					injectHtml += `<div class="form-group">
                        <label for="${k}">${v.label || ""}</label>
                            <input type="checkbox" name="${flag}" ${flagValue ? "checked" : ""}>${notes}
                    </div>`;
					break;
				case "select":
					injectHtml += `<div class="form-group">
                        <label for="${k}">${v.label || ""}</label>
                            <select name="${flag}">`;
					for (const [i, j] of Object.entries(v.options)) {
						injectHtml += `<option value="${i}" ${flagValue === i ? "selected" : ""}>${j}</option>`;
					}
					injectHtml += `</select>${notes}
                    </div>`;
					break;
				case "range":
					injectHtml += `<div class="form-group">
                        <label for="${k}">${v.label || ""}</label>
                        <div class="form-fields">
                            <input type="range" name="${flag}" value="${flagValue}" min="${v.min}" max="${v.max}" step="${v.step ?? 1}">
                            <span class="range-value">${flagValue}</span>${notes}
                        </div>
                    </div>`;
					break;
				case "color":
					injectHtml += `<div class="form-group">
                        <label for="${k}">${v.label || ""}</label>
                        <div class="form-fields">
                            <input class="color" type="text" name="${flag}" value="${flagValue}">
                            <input type="color" data-edit="${flag}" value="${flagValue}">
                        </div>
                        ${notes}
                    </div>`;
					break;
				case "custom":
					injectHtml += v.html;
					break;
			}
			if (elemData.type?.includes("filepicker")) {
				const fpType = elemData.type.split(".")[1] || "imagevideo";
				injectHtml += `<div class="form-group">
                <label for="${k}">${v.label || ""}</label>
                <div class="form-fields">
                    <button type="button" class="file-picker" data-extras="${
	elemData.fpTypes ? elemData.fpTypes.join(",") : ""
}" data-type="${fpType}" data-target="${flag}" title="Browse Files" tabindex="-1">
                        <i class="fas fa-file-import fa-fw"></i>
                    </button>
                    <input class="image" type="text" name="${flag}" placeholder="${v.placeholder || ""}" value="${flagValue}">
                </div>${notes}
            </div>`;
			}
		}
		injectHtml = $(injectHtml);
		injectHtml.on("click", ".file-picker", this.fpTypes, _bindFilePicker);
		injectHtml.on("change", "input[type=\"color\"]", _colorChange);
		if (data.tab) {
			const injectTab = createTab(data.tab.name, data.tab.label, data.tab.icon).append(injectHtml);
			injectPoint.after(injectTab);
			app?.setPosition({ height: "auto", width: data.tab ? app.options.width + tabSize : "auto" });
			return injectHtml;
		}
		injectPoint.after(injectHtml);
		if (app) app?.setPosition({ height: "auto", width: data.tab ? app.options.width + tabSize : "auto" });
		return injectHtml;

		function createTab(name, label, icon) {
			/* let tabs = html.find(".sheet-tabs").last();
            if(!tabs.length) tabs = html.find(`nav[data-group="main"]`);*/
			const tabs = html.find(".sheet-tabs").first().find(".item").last();
			const tab = `<a class="item" data-tab="${name}"><i class="${icon}"></i> ${label}</a>`;
			tabs.after(tab);
			const tabContainer = `<div class="tab" data-tab="${name}"></div>`;
			return $(tabContainer);
		}

		function getDefaultFlag(inputType) {
			switch (inputType) {
				case "number":
					return 0;
				case "checkbox":
					return false;
			}
			return "";
		}

		function _colorChange(e) {
			const input = $(e.target);
			const edit = input.data("edit");
			const value = input.val();
			injectHtml.find(`input[name="${edit}"]`).val(value);
		}

		function _bindFilePicker(event) {
			event.preventDefault();
			const button = event.currentTarget;
			const input = $(button).closest(".form-fields").find("input") || null;
			const extraExt = button.dataset.extras ? button.dataset.extras.split(",") : [];
			const options = {
				field: input[0],
				type: button.dataset.type,
				current: input.val() || null,
				button: button,
			};
			const fp = new FilePicker(options);
			fp.extensions ? fp.extensions.push(...extraExt) : (fp.extensions = extraExt);
			return fp.browse();
		}
	},
	quickInject: function quickInject(injectData, data) {
		injectData = Array.isArray(injectData) ? injectData : [injectData];
		for (const doc of injectData) {
			let newData = data;
			if (doc.inject) {
				newData = JSON.parse(JSON.stringify(data));
				data.inject = doc.inject;
			}
			Hooks.on(`render${doc.documentName}Config`, (app, html) => {
				injectConfig.inject(app, html, newData);
			});
		}
	},
	_generateTabStruct: function _generateTabStruct(app, html, data, object) {
		const isTabs = html.find(".sheet-tabs").length;
		const useTabs = data.tab;
		if (isTabs || !useTabs) return;
		const tabSize = data.tab?.width || 100;
		const layer = app?.object?.layer?.options?.name;
		const icon = $(".main-controls").find(`li[data-canvas-layer="${layer}"]`).find("i").attr("class");

		const $tabs = $(`<nav class="sheet-tabs tabs">
        <a class="item active" data-tab="basic"><i class="${icon}"></i> ${game.i18n.localize("LIGHT.HeaderBasic")}</a>
        </nav>
        <div class="tab active" data-tab="basic"></div>`);
		// move all content of form into tab
		const form = html.find("form").first();
		form.children().each((i, e) => {
			$($tabs[2]).append(e);
		});

		form.append($tabs);
		const submitButton = html.find("button[type='submit']").first();
		form.append(submitButton);

		html.on("click", ".item", (e) => {
			html.find(".item").removeClass("active");
			$(e.currentTarget).addClass("active");
			html.find(".tab").removeClass("active");
			html.find(`[data-tab="${e.currentTarget.dataset.tab}"]`).addClass("active");
			app.setPosition({ height: "auto", width: data.tab ? app.options.width + tabSize : "auto" });
		});
	},
};

const MODULE_ID = "about-face";
const IndicatorMode = {
	OFF: 0,
	HOVER: 1,
	ALWAYS: 2,
};
const facingOptions = {
	global: {},
	none: {},
	rotate: {
		right: "about-face.options.facing-direction.choices.right",
		left: "about-face.options.facing-direction.choices.left",
		down: "about-face.options.facing-direction.choices.down",
		up: "about-face.options.facing-direction.choices.up",
	},
	"flip-h": {
		right: "about-face.options.facing-direction.choices.right",
		left: "about-face.options.facing-direction.choices.left",
	},
	"flip-v": {
		down: "about-face.options.facing-direction.choices.down",
		up: "about-face.options.facing-direction.choices.up",
	},
};

function getAllTokens() {
	const tokens = [];
	canvas.scene.tokens.forEach((tokenDocument) => {
		if (tokenDocument.object) tokens.push(tokenDocument.object);
	});
	return tokens;
}

function registerSettings() {
	const SYSTEM_DEFAULTS = {};
	let system = /gurps/.exec(game.system.id);
	if (system) {
		switch (system[0]) {
			case "gurps":
				SYSTEM_DEFAULTS.lockArrowToFace = true;
				SYSTEM_DEFAULTS.lockRotation = true;
				SYSTEM_DEFAULTS["flip-or-rotate"] = "rotate";
				break;
			default:
				console.error("About Face | Somehow, this happened.");
		}
	}

	game.settings.register(MODULE_ID, "arrowColor", {
		name: "about-face.options.arrowColor.name",
		hint: "about-face.options.arrowColor.hint",
		scope: "world",
		config: true,
		default: "#000000",
		type: String,
		onChange: (value) => {
			game.aboutFace.indicatorColor = value;
			if (canvas === null) return;
			const tokens = getAllTokens();
			for (const token of tokens) {
				if (token.aboutFaceIndicator) {
					token.aboutFaceIndicator.destroy();
					game.aboutFace.drawAboutFaceIndicator(token);
				}
			}
		},
	});

	game.settings.register(MODULE_ID, "arrowDistance", {
		name: "about-face.options.arrowDistance.name",
		hint: "about-face.options.arrowDistance.hint",
		scope: "world",
		config: true,
		default: 1.4,
		type: Number,
		range: {
			min: 1.0,
			max: 1.4,
			step: 0.05,
		},
		onChange: (value) => {
			game.aboutFace.indicatorDistance = value;
			if (canvas === null) return;
			const tokens = getAllTokens();
			for (const token of tokens) {
				if (token.aboutFaceIndicator) game.aboutFace.drawAboutFaceIndicator(token);
			}
		},
	});

	game.settings.register(MODULE_ID, "indicator-state", {
		name: "about-face.options.enable-indicator.name",
		hint: "about-face.options.enable-indicator.hint",
		scope: "world",
		config: true,
		default: 2,
		type: Number,
		choices: {
			0: "about-face.options.enable-indicator.choices.0",
			1: "about-face.options.enable-indicator.choices.1",
			2: "about-face.options.enable-indicator.choices.2",
		},
		onChange: (value) => {
			value = Number(value);
			if (value === IndicatorMode.HOVER) {
				Hooks.on("hoverToken", tokenHover);
				Hooks.on("highlightObjects", highlightObjects);
			} else {
				Hooks.off("hoverToken", tokenHover);
				Hooks.off("highlightObjects", highlightObjects);
			}
			toggleAllIndicators(value === IndicatorMode.ALWAYS);
		},
	});

	game.settings.register(MODULE_ID, "indicator-state-pc", {
		name: "about-face.options.enable-indicator-pc.name",
		hint: "about-face.options.enable-indicator-pc.hint",
		scope: "world",
		config: true,
		default: 2,
		type: Number,
		choices: {
			0: "about-face.options.enable-indicator.choices.0",
			1: "about-face.options.enable-indicator.choices.1",
			2: "about-face.options.enable-indicator.choices.2",
		},
		onChange: (value) => {
			value = Number(value);
			if (value === IndicatorMode.HOVER) {
				Hooks.on("hoverToken", tokenHover);
				Hooks.on("highlightObjects", highlightObjects);
			} else {
				Hooks.off("hoverToken", tokenHover);
				Hooks.off("highlightObjects", highlightObjects);
			}
			toggleAllIndicators(value === IndicatorMode.ALWAYS, true);
		},
	});

	game.settings.register(MODULE_ID, "sprite-type", {
		name: "about-face.options.indicator-sprite.name",
		hint: "about-face.options.indicator-sprite.hint",
		scope: "world",
		config: true,
		default: 1.0,
		type: Number,
		range: {
			min: 0.5,
			max: 2.0,
			step: 0.05,
		},
		onChange: (value) => {
			game.aboutFace.indicatorSize = value;
			if (canvas === null) return;
			const tokens = getAllTokens();
			for (const token of tokens) {
				game.aboutFace.drawAboutFaceIndicator(token);
			}
		},
	});

	game.settings.register(MODULE_ID, "indicatorDrawingType", {
		name: "about-face.options.indicatorDrawingType.name",
		hint: "about-face.options.indicatorDrawingType.hint",
		scope: "world",
		config: true,
		default: 0,
		type: Number,
		choices: {
			0: game.i18n.localize("about-face.options.indicatorDrawingType.options.arrow"),
			1: game.i18n.localize("about-face.options.indicatorDrawingType.options.line"),
		},
		requiresReload: true,
	});

	game.settings.register(MODULE_ID, "hideIndicatorOnDead", {
		name: "about-face.options.hideIndicatorOnDead.name",
		hint: "about-face.options.hideIndicatorOnDead.hint",
		scope: "world",
		config: true,
		default: false,
		type: Boolean,
		onChange: (value) => {
			game.aboutFace.hideIndicatorOnDead = value;
			if (canvas === null) return;
			const tokens = getAllTokens();
			for (const token of tokens) {
				game.aboutFace.drawAboutFaceIndicator(token);
			}
		},
	});

	game.settings.register(MODULE_ID, "lockArrowRotation", {
		name: "about-face.options.lockArrowRotation.name",
		hint: "about-face.options.lockArrowRotation.hint",
		scope: "world",
		config: true,
		default: false,
		type: Boolean,
		onChange: (value) => {
			new Dialog({
				title: game.i18n.localize("about-face.options.lockArrowRotation.name"),
				content: game.i18n.localize("about-face.options.changeEverySceneDialog"),
				buttons: {
					yes: {
						label: game.i18n.localize("Yes"),
						callback: (html) => {
							game.scenes.updateAll({ flags: { [MODULE_ID]: { lockArrowRotation: value } } });
						},
					},
					no: {
						label: game.i18n.localize("No"),
					},
				},
			}).render(true);
		},
	});
	game.settings.register(MODULE_ID, "lockArrowToFace", {
		name: "about-face.options.lockArrowToFace.name",
		hint: "about-face.options.lockArrowToFace.hint",
		scope: "world",
		config: true,
		default: SYSTEM_DEFAULTS.lockArrowToFace || false,
		type: Boolean,
	});

	game.settings.register(MODULE_ID, "disableAnimations", {
		name: "about-face.options.disableAnimations.name",
		hint: "about-face.options.disableAnimations.hint",
		scope: "world",
		config: true,
		default: false,
		type: Boolean,
		onChange: (value) => {
			game.aboutFace._prepareAnimation(value);
		},
	});

	game.settings.register(MODULE_ID, "lockRotation", {
		name: "about-face.options.lockRotation.name",
		hint: "about-face.options.lockRotation.hint",
		scope: "world",
		config: true,
		default: SYSTEM_DEFAULTS.lockRotation || false,
		type: Boolean,
		onChange: (value) => {
			new Dialog({
				title: game.i18n.localize("about-face.options.lockRotation.name"),
				content: game.i18n.localize("about-face.options.changeEverySceneDialog"),
				buttons: {
					yes: {
						label: game.i18n.localize("Yes"),
						callback: (html) => {
							game.scenes.updateAll({ flags: { [MODULE_ID]: { lockRotation: value } } });
						},
					},
					no: {
						label: game.i18n.localize("No"),
					},
				},
			}).render(true);
		},
	});

	game.settings.register(MODULE_ID, "flip-or-rotate", {
		name: "about-face.options.flip-or-rotate.name",
		hint: "about-face.options.flip-or-rotate.hint",
		scope: "world",
		config: true,
		default: SYSTEM_DEFAULTS["flip-or-rotate"] || "flip-h",
		type: String,
		choices: {
			rotate: "about-face.options.flip-or-rotate.choices.rotate",
			"flip-h": "about-face.options.flip-or-rotate.choices.flip-h",
			"flip-v": "about-face.options.flip-or-rotate.choices.flip-v",
		},
	});
	game.settings.register(MODULE_ID, "facing-direction", {
		name: "about-face.options.facing-direction.name",
		hint: "about-face.options.facing-direction.hint",
		scope: "world",
		config: true,
		default: "right",
		type: String,
		choices: {
			right: "about-face.options.facing-direction.choices.right",
			left: "about-face.options.facing-direction.choices.left",
		},
	});
	game.settings.register(MODULE_ID, "lockVisionToRotation", {
		name: "about-face.options.lockVisionToRotation.name",
		hint: "about-face.options.lockVisionToRotation.hint",
		scope: "world",
		config: true,
		default: true,
		type: Boolean,
	});
}

/**
 * Handler called when token configuration window is opened. Injects custom form html and deals
 * with updating token.
 * @category GMOnly
 * @function
 * @async
 * @param {TokenConfig} tokenConfig
 * @param {JQuery} html
 */
async function renderSettingsConfigHandler(tokenConfig, html) {
	const lockArrowRotation = game.settings.get(MODULE_ID, "lockArrowRotation");
	const lockArrowRotationCheckbox = html.find('input[name="about-face.lockArrowRotation"]');
	const lockArrowToFaceCheckbox = html.find('input[name="about-face.lockArrowToFace"]');
	disableCheckbox(lockArrowToFaceCheckbox, lockArrowRotation);

	lockArrowRotationCheckbox.on("change", (event) => {
		disableCheckbox(lockArrowToFaceCheckbox, event.target.checked);
	});

	const flipOrRotate = game.settings.get(MODULE_ID, "flip-or-rotate");
	const flipOrRotateSelect = html.find('select[name="about-face.flip-or-rotate"]');
	const flipDirectionSelect = html.find('select[name="about-face.facing-direction"]');
	replaceSelectChoices(flipDirectionSelect, facingOptions[flipOrRotate]);

	const lockVisionToRotationCheckbox = html.find('input[name="about-face.lockVisionToRotation"]');

	disableCheckbox(lockVisionToRotationCheckbox, flipOrRotate !== "rotate");
	flipOrRotateSelect.on("change", (event) => {
		const facingDirections = facingOptions[event.target.value];
		replaceSelectChoices(flipDirectionSelect, facingDirections);
		disableCheckbox(lockVisionToRotationCheckbox, event.target.value !== "rotate");
	});

	// Create color picker
	const arrowColorInput = html.find('input[name="about-face.arrowColor"]');
	if (arrowColorInput.length) colorPicker("about-face.arrowColor", html, game.settings.get(MODULE_ID, "arrowColor"));
}

function disableCheckbox(checkbox, boolean) {
	checkbox.prop("disabled", boolean);
}

function replaceSelectChoices(select, choices) {
	const facing = game.settings.get(MODULE_ID, "facing-direction");
	select.empty();
	let hasGlobal = false;
	for (const [key, value] of Object.entries(choices)) {
		if (key === "global") {
			hasGlobal = true;
			select.append(
				$("<option></option>").attr("value", key).attr("selected", true).text(game.i18n.localize(value))
			);
		} else {
			select.append(
				$("<option></option>")
					.attr("value", key)
					.attr("selected", !hasGlobal && facing === key)
					.text(game.i18n.localize(value))
			);
		}
	}
}

/**
 * Handler called when token configuration window is opened. Injects custom form html and deals
 * with updating token.
 * @category GMOnly
 * @function
 * @async
 * @param {TokenConfig} tokenConfig
 * @param {JQuery} html
 */
async function renderTokenConfigHandler(tokenConfig, html) {
	injectConfig.inject(
		tokenConfig,
		html,
		{
			moduleId: MODULE_ID,
			tab: {
				name: MODULE_ID,
				label: game.i18n.localize("about-face.options.facing"),
				icon: "fas fa-caret-down fa-fw",
			},
		},
		tokenConfig.object
	);
	const posTab = html.find(`.tab[data-tab="${MODULE_ID}"]`);

	// const flipOrRotate = tokenConfig.object.getFlag(MODULE_ID, "flipOrRotate") || "global";
	if (tokenConfig.options.sheetConfig) {
		var indicatorDisabled = tokenConfig.object.getFlag(MODULE_ID, "indicatorDisabled") ? "checked" : "";
		var flipOrRotate = tokenConfig.object.getFlag(MODULE_ID, "flipOrRotate") || "global";
		var facingDirection = tokenConfig.object.getFlag(MODULE_ID, "facingDirection") || "global";
		var rotationOffset = tokenConfig.object.getFlag(MODULE_ID, "rotationOffset") || "0";
	} else {
		indicatorDisabled = tokenConfig.token.flags?.[MODULE_ID]?.indicatorDisabled ? "checked" : "";
		flipOrRotate = tokenConfig.token.flags?.[MODULE_ID]?.flipOrRotate || "global";
		facingDirection = tokenConfig.token.flags?.[MODULE_ID]?.facingDirection || "global";
		rotationOffset = tokenConfig.token.flags?.[MODULE_ID]?.rotationOffset || "0";
	}
	const flipOrRotateSetting = game.settings.get(MODULE_ID, "flip-or-rotate");
	const flipOrRotates = {
		global: `${game.i18n.localize("about-face.options.flip-or-rotate.choices.global")} (${game.i18n.localize(
			`about-face.options.flip-or-rotate.choices.${flipOrRotateSetting}`
		)})`,
		...game.settings.settings.get("about-face.flip-or-rotate").choices,
	};
	const facingDirectionSetting = game.settings.get(MODULE_ID, "facing-direction");
	const facingDirections = {
		global: `${game.i18n.localize("about-face.options.flip-or-rotate.choices.global")} (${game.i18n.localize(
			`about-face.options.facing-direction.choices.${facingDirectionSetting}`
		)})`,
		...facingOptions[flipOrRotateSetting],
	};
	let data = {
		indicatorDisabled: indicatorDisabled,
		flipOrRotates: flipOrRotates,
		flipOrRotate: flipOrRotate,
		facingDirection: facingDirection,
		facingDirections: facingDirections,
		rotationOffset: rotationOffset,
	};

	const insertHTML = await renderTemplate(`modules/${MODULE_ID}/templates/token-config.html`, data);
	posTab.append(insertHTML);

	const selectFlipOrRotate = posTab.find(".token-config-select-flip-or-rotate");
	const selectFacingDirection = posTab.find(".token-config-select-facing-direction");

	selectFlipOrRotate.on("change", (event) => {
		const flipOrRotate = event.target.value !== "global" ? event.target.value : flipOrRotateSetting;
		const facingDirections = {};
		if (event.target.value === "global") {
			facingDirections.global = `${game.i18n.localize(
				"about-face.options.flip-or-rotate.choices.global"
			)} (${game.i18n.localize(`about-face.options.facing-direction.choices.${facingDirectionSetting}`)})`;
		}
		foundry.utils.mergeObject(facingDirections, facingOptions[flipOrRotate]);
		replaceSelectChoices(selectFacingDirection, facingDirections);
	});
}

function renderSceneConfigHandler(app, html) {
	const data = {
		moduleId: MODULE_ID,
		tab: {
			name: MODULE_ID,
			label: "About Face",
			icon: "fas fa-caret-down fa-fw",
		},
		sceneEnabled: {
			type: "checkbox",
			label: game.i18n.localize("about-face.sceneConfig.scene-enabled.name"),
			notes: game.i18n.localize("about-face.sceneConfig.scene-enabled.hint"),
			default: app.object?.flags?.[MODULE_ID]?.sceneEnabled ?? true,
		},
		lockRotation: {
			type: "checkbox",
			label: game.i18n.localize("about-face.sceneConfig.lockRotation.name"),
			notes: game.i18n.localize("about-face.sceneConfig.lockRotation.hint"),
			default: app.object?.flags?.[MODULE_ID]?.lockRotation ?? game.settings.get(MODULE_ID, "lockRotation"),
		},
		lockRotationButton: {
			type: "custom",
			html: `<button type="button" id="lockRotationButton">${game.i18n.localize(
				"about-face.sceneConfig.apply"
			)}</button>`,
		},
		lockArrowRotation: {
			type: "checkbox",
			label: game.i18n.localize("about-face.sceneConfig.lockArrowRotation.name"),
			notes: game.i18n.localize("about-face.sceneConfig.lockArrowRotation.hint"),
			default:
				app.object?.flags?.[MODULE_ID]?.lockArrowRotation ?? game.settings.get(MODULE_ID, "lockArrowRotation"),
		},
		lockArrowRotationButton: {
			type: "custom",
			html: `<button type="button" id="lockArrowRotationButton">${game.i18n.localize(
				"about-face.sceneConfig.apply"
			)}</button>`,
		},
	};
	injectConfig.inject(app, html, data, app.object);
}

async function asyncRenderSceneConfigHandler(app, html) {
	const lockRotationButton = html.find("button[id='lockRotationButton']");
	lockRotationButton.on("click", () => {
		const lockRotationCheckbox = html.find('input[name="flags.about-face.lockRotation"]');
		const state = lockRotationCheckbox[0].checked;
		const updates = [];
		canvas.scene.tokens.forEach((token) => {
			if (token.lockRotation !== state) {
				updates.push({
					_id: token.id,
					lockRotation: state,
				});
			}
		});
		canvas.scene.updateEmbeddedDocuments("Token", updates);
	});
	const lockArrowRotationButton = html.find("button[id='lockArrowRotationButton']");
	lockArrowRotationButton.on("click", () => {
		const lockArrowRotationCheckbox = html.find('input[name="flags.about-face.lockArrowRotation"]');
		const state = lockArrowRotationCheckbox[0].checked;
		const updates = [];
		canvas.scene.tokens.forEach((token) => {
			if ("token.flags.about-face.lockArrowRotation" !== state) {
				updates.push({
					_id: token.id,
					flags: {
						"about-face": { lockArrowRotation: state },
					},
				});
			}
		});
		canvas.scene.updateEmbeddedDocuments("Token", updates);
	});
}

function toggleAllIndicators(state, playerOwner = false) {
	if (canvas === null) return;
	const tokens = getAllTokens();
	tokens.forEach((token) => {
		if (token.actor.hasPlayerOwner === playerOwner && token.aboutFaceIndicator) {
			token.aboutFaceIndicator.graphics.visible = state;
		}
	});
}

function tokenHover(token, hovered) {
	if (hovered) {
		game.aboutFace.drawAboutFaceIndicator(token);
	} else if (token.aboutFaceIndicator) {
		token.aboutFaceIndicator.graphics.visible = false;
	}
}
function highlightObjects(highlighted) {
	if (canvas.scene?.flags?.[MODULE_ID].sceneEnabled) {
		canvas.scene.tokens.forEach((tokenDocument) => {
			if (highlighted) {
				game.aboutFace.drawAboutFaceIndicator(tokenDocument.object);
			} else if (tokenDocument.object.aboutFaceIndicator) {
				tokenDocument.object.aboutFaceIndicator.graphics.visible = false;
			}
		});
	}
}

class AboutFace {
	constructor() {
		this.indicatorColor = game.settings.get(MODULE_ID, "arrowColor");
		this.indicatorDistance = game.settings.get(MODULE_ID, "arrowDistance");
		this.hideIndicatorOnDead = game.settings.get("about-face", "hideIndicatorOnDead");
		this.indicatorDrawingType = game.settings.get("about-face", "indicatorDrawingType");
		this.indicatorSize = game.settings.get("about-face", "sprite-type");
		this._tokenRotation = false;
		if (game.settings.get("about-face", "disableAnimations")) this._prepareAnimation();
	}

	get tokenRotation() {
		return this._tokenRotation;
	}

	set tokenRotation(value) {
		this._tokenRotation = value;
	}

	_prepareAnimation(value = true) {
		if (value) {
			libWrapper.register(
				MODULE_ID,
				"CONFIG.Token.objectClass.prototype._getAnimationDuration",
				(from, to, { movementSpeed = 6 } = {}) => {
					let duration = 0;
					const dx = from.x - (to.x ?? from.x);
					const dy = from.y - (to.y ?? from.y);
					if (dx || dy) duration = Math.max(
						duration,
						(Math.hypot(dx, dy) / canvas.dimensions.size / movementSpeed) * 1000
					);
					const dr = ((Math.abs(from.rotation - (to.rotation ?? from.rotation)) + 180) % 360) - 180;
					if (dr && !(dx || dy)) duration = Math.max(duration, (Math.abs(dr) / (movementSpeed * 60)) * 1000);
					if (!duration) duration = 1000; // The default animation duration is 1 second
					return duration;
				},
				"OVERRIDE"
			);
			libWrapper.register(
				MODULE_ID,
				"CONFIG.Token.objectClass.prototype._prepareAnimation",
				(wrapped, from, changes, context, options = {}) => {
					if ("x" in changes || "y" in changes) {
						if ("rotation" in changes) delete changes.rotation;
						if (changes.texture?.scaleX) delete changes.texture.scaleX;
						if (changes.texture?.scaleY) delete changes.texture.scaleY;
					}
					return wrapped(from, changes, context, options);
				},
				"WRAPPER"
			);
		} else {
			libWrapper.unregister(MODULE_ID, "CONFIG.Token.objectClass.prototype._getAnimationDuration");
			libWrapper.unregister(MODULE_ID, "CONFIG.Token.objectClass.prototype._prepareAnimation");
		}
	}

	drawAboutFaceIndicator(token) {
		if (!canvas.scene.getFlag(MODULE_ID, "sceneEnabled")) {
			if (token.aboutFaceIndicator) token.aboutFaceIndicator.graphics.visible = false;
			return;
		}
		const deadIcon = CONFIG.statusEffects.find((x) => x.id === "dead")?.icon;
		const isDead = token.actor?.effects.some((el) => el.statuses.has("dead") || el.img === deadIcon);
		if (this.hideIndicatorOnDead && isDead) {
			if (token.aboutFaceIndicator && !token.aboutFaceIndicator?._destroyed) {
				token.aboutFaceIndicator.graphics.visible = false;
			}
			return;
		}
		try {
			// get the rotation of the token
			let tokenDirection = token.document.flags[MODULE_ID]?.direction
				?? getIndicatorDirection(token.document) ?? 90;

			// Calculate indicator's distance
			const indicatorDistance = this.indicatorDistance;
			const maxTokenSize = Math.max(token.w, token.h);
			const distance = (maxTokenSize / 2) * indicatorDistance;

			// Calculate indicator's scale
			const tokenSize = Math.max(token.document.width, token.document.height);
			const tokenScale = Math.abs(token.document.texture.scaleX) + Math.abs(token.document.texture.scaleY);
			const indicatorSize = this.indicatorSize || 1;
			const scale = ((tokenSize * tokenScale) / 2) * indicatorSize;

			// Create or update the about face indicator
			// updateAboutFaceIndicator(token, tokenDirection, distance, scale);
			const { w: width, h: height } = token;
			if (!token.aboutFaceIndicator || token.aboutFaceIndicator._destroyed) {
				const container = new PIXI.Container({ name: "aboutFaceIndicator", width, height });
				container.name = "aboutFaceIndicator";
				container.width = width;
				container.height = height;
				container.x = width / 2;
				container.y = height / 2;
				const graphics = new PIXI.Graphics();
				// draw an arrow indicator
				// drawArrow(graphics);
				const color = `0x${this.indicatorColor.substring(1, 7)}` || "";
				graphics.beginFill(color, 0.5).lineStyle(2, color, 1).moveTo(0, 0);
				if (this.indicatorDrawingType === 0) {
					graphics.lineTo(0, -10).lineTo(10, 0).lineTo(0, 10).lineTo(0, 0).closePath().endFill();
				} else if (this.indicatorDrawingType === 1) {
					graphics.lineTo(-10, -20).lineTo(0, 0).lineTo(-10, 20).lineTo(0, 0).closePath().endFill();
				}
				// place the arrow in the correct position
				container.angle = tokenDirection;
				graphics.x = distance;
				graphics.scale.set(scale, scale);
				// add the graphics to the container
				container.addChild(graphics);
				container.graphics = graphics;
				token.aboutFaceIndicator = container;
				// add the container to the token
				token.addChild(container);
			} else {
				let container = token.aboutFaceIndicator;
				let graphics = container.graphics;
				container.x = width / 2;
				container.y = height / 2;
				graphics.x = distance;
				graphics.scale.set(scale, scale);
				// update the rotation of the arrow
				container.angle = tokenDirection;
			}

			// Set the visibility of the indicator based on the current indicator mode
			const indicatorState = token?.actor?.hasPlayerOwner
				? game.settings.get(MODULE_ID, "indicator-state-pc")
				: game.settings.get(MODULE_ID, "indicator-state");
			const indicatorDisabled = token.document.getFlag(MODULE_ID, "indicatorDisabled");

			if (indicatorState === IndicatorMode.OFF || indicatorDisabled) {
				token.aboutFaceIndicator.graphics.visible = false;
			} else if (indicatorState === IndicatorMode.HOVER) token.aboutFaceIndicator.graphics.visible = token.hover;
			else if (indicatorState === IndicatorMode.ALWAYS) token.aboutFaceIndicator.graphics.visible = true;
		} catch(error) {
			console.error(
				`About Face | Error drawing the indicator for token ${token.name} (ID: ${token.id}, Type: ${
					token.document?.actor?.type ?? null
				})`,
				error
			);
		}
	}
}

// HOOKS

function onPreCreateToken(tokenDocument, data, options, userId) {
	const updates = { flags: { [MODULE_ID]: {} } };
	const facingDirection =
		tokenDocument.flags?.[MODULE_ID]?.facingDirection ?? game.settings.get(MODULE_ID, "facing-direction");
	if (canvas.scene.getFlag(MODULE_ID, "lockRotation")) {
		updates.lockRotation = true;
	}
	if (facingDirection) {
		const flipMode = game.settings.get(MODULE_ID, "flip-or-rotate");
		const gridType = getGridType();
		if (gridType === 0 || (gridType === 1 && flipMode === "flip-h") || (gridType === 2 && flipMode === "flip-v")) {
			const TokenDirections = {
				down: 90,
				right: 360,
				up: 270,
				left: 180,
			};
			if (tokenDocument.flags?.[MODULE_ID]?.direction === undefined) {
				updates.flags[MODULE_ID].direction = TokenDirections[facingDirection];
			}
		}
	}
	if (Object.keys(updates).length) tokenDocument.updateSource(updates);
}

function onPreUpdateToken(tokenDocument, updates, options, userId) {
	if (!canvas.scene.getFlag(MODULE_ID, "sceneEnabled")) {
		return;
	}

	if (
		game.modules.get("multilevel-tokens")?.active
		&& !game.multilevel._isReplicatedToken(tokenDocument)
		&& options?.mlt_bypass
	) {
		return;
	}

	if (
		tokenDocument.x === updates.x
		&& tokenDocument.y === updates.y
		&& (
			!("rotation" in updates)
			|| tokenDocument.rotation === updates.rotation
		)
	) {
		return;
	}

	let position = {};
	// store the direction in the token data

	let tokenDirection;
	const { x, y, rotation } = updates;
	const { x: tokenX, y: tokenY } = tokenDocument;

	if (rotation !== undefined) {
		tokenDirection = rotation + 90;
		foundry.utils.setProperty(updates, `flags.${MODULE_ID}.direction`, tokenDirection);
		if (!options.animation) {
			options.animation = { duration: 1000 / 6 };
		} else {
			options.animation.duration = 1000 / 6;
		}
	} else if (
		!game.aboutFace.tokenRotation
		&& (Number.isNumeric(x) || Number.isNumeric(y))
		&& !canvas.scene.getFlag(MODULE_ID, "lockArrowRotation")
	) {
		// get previous and new positions
		const prevPos = { x: tokenX, y: tokenY };
		const newPos = { x: x ?? tokenX, y: y ?? tokenY };
		// get the direction in degrees of the movement
		let diffY = newPos.y - prevPos.y;
		let diffX = newPos.x - prevPos.x;
		tokenDirection = (Math.atan2(diffY, diffX) * 180) / Math.PI;

		if (canvas.grid.type && game.settings.get(MODULE_ID, "lockArrowToFace")) {
			const directions = [
				[45, 90, 135, 180, 225, 270, 315, 360], // Square
				[0, 60, 120, 180, 240, 300, 360], // Hex Rows
				[30, 90, 150, 210, 270, 330, 390], // Hex Columns
			];
			const gridType = getGridType();
			const facings = directions[gridType];
			if (facings && facings.length) {
				// convert negative dirs into a range from 0-360
				let normalizedDir = ((tokenDirection % 360) + 360) % 360; // Math.round(tokenDirection < 0 ? 360 + tokenDirection : tokenDirection);
				// find the largest normalized angle
				let secondAngle = facings.reduceRight(
					(prev, curr) => (curr < prev && curr > normalizedDir ? curr : prev)
				); // facings.find((e) => e > normalizedDir);
				// and assume the facing is 60 degrees (hexes) or 45 (square) to the counter clockwise
				tokenDirection = gridType ? secondAngle - 60 : secondAngle - 45;
				// unless the largest angle was closer
				if (secondAngle - normalizedDir < normalizedDir - tokenDirection) tokenDirection = secondAngle;
				// return tokenDirection to the range 180 to -180
				if (tokenDirection > 180) tokenDirection -= 360;
			}
		}
		foundry.utils.setProperty(updates, `flags.${MODULE_ID}.direction`, tokenDirection);
		foundry.utils.setProperty(updates, `flags.${MODULE_ID}.prevPos`, prevPos);
		position = { x: diffX, y: diffY };
	}

	const { texture, lockRotation, flags, sight } = tokenDocument;
	const flipOrRotate = getTokenFlipOrRotate(tokenDocument);

	if (flipOrRotate !== "rotate") {
		const [mirrorKey, mirrorVal] = getMirror(tokenDocument, position);
		if ((texture[mirrorKey] < 0 && !mirrorVal) || (texture[mirrorKey] > 0 && mirrorVal)) {
			const source = tokenDocument.toObject();
			updates[`texture.${mirrorKey}`] = source.texture[mirrorKey] * -1;
		}
	}

	// update the rotation of the token
	if (tokenDirection === undefined) return;

	// Determine if a rotation update is needed
	const hasRotationUpdate = "rotation" in updates;
	const isRotationAction = flipOrRotate === "rotate";
	const isRotationLocked = lockRotation && game.settings.get(MODULE_ID, "lockVisionToRotation");

	// Check conditions for rotation and sight angle updates
	const shouldUpdateRotation = hasRotationUpdate || isRotationAction !== isRotationLocked;
	const hasSightAngle = sight.enabled && sight.angle !== 360;
	const shouldUpdateSightAngle = !isRotationLocked && hasSightAngle;

	// If any update is needed, calculate the rotation offset and update the rotation
	if (shouldUpdateRotation || shouldUpdateSightAngle) {
		const rotationOffset = flags[MODULE_ID]?.rotationOffset ?? 0;
		updates.rotation = tokenDirection - 90 + rotationOffset;
	}
}

// HELPERS

function getGridType() {
	return Math.floor(canvas.grid.type / 2);
}

function getIndicatorDirection(tokenDocument) {
	const IndicatorDirections = {
		up: -90,
		right: 0,
		down: 90,
		left: 180,
	};
	const direction =
		tokenDocument.getFlag(MODULE_ID, "facingDirection") || game.settings.get(MODULE_ID, "facing-direction");
	return IndicatorDirections[direction];
}

function getTokenFlipOrRotate(tokenDocument) {
	const tokenFlipOrRotate = tokenDocument.getFlag(MODULE_ID, "flipOrRotate") || "global";
	return tokenFlipOrRotate !== "global" ? tokenFlipOrRotate : game.settings.get(MODULE_ID, "flip-or-rotate");
}

/**
 *
 * @param {TokenDocument} tokenDocument
 * @param {Object} position
 * @returns {Array}
 */
function getMirror(tokenDocument, position = {}) {
	if (!Object.keys(position).length) {
		// Taken from ClientKeybindings._handleMovement
		// Define movement offsets and get moved directions
		const directions = game.keybindings.moveKeys;
		let dx = 0;
		let dy = 0;

		// Assign movement offsets
		if (directions.has(ClientKeybindings.MOVEMENT_DIRECTIONS.LEFT)) dx -= 1;
		else if (directions.has(ClientKeybindings.MOVEMENT_DIRECTIONS.RIGHT)) dx += 1;
		if (directions.has(ClientKeybindings.MOVEMENT_DIRECTIONS.UP)) dy -= 1;
		else if (directions.has(ClientKeybindings.MOVEMENT_DIRECTIONS.DOWN)) dy += 1;

		position = { x: dx, y: dy };
	}
	const { x, y } = position;
	const tokenFacingDirection = tokenDocument.getFlag(MODULE_ID, "facingDirection") || "global";
	const facingDirection =
		tokenFacingDirection === "global" ? game.settings.get(MODULE_ID, "facing-direction") : tokenFacingDirection;
	const mirrorX = "scaleX";
	const mirrorY = "scaleY";
	if (facingDirection === "right") {
		if (x < 0) return [mirrorX, true];
		if (x > 0) return [mirrorX, false];
	} else if (facingDirection === "left") {
		if (x < 0) return [mirrorX, false];
		if (x > 0) return [mirrorX, true];
	} else if (facingDirection === "up") {
		if (y < 0) return [mirrorY, false];
		if (y > 0) return [mirrorY, true];
	} else if (facingDirection === "down") {
		if (y < 0) return [mirrorY, true];
		if (y > 0) return [mirrorY, false];
	}
	return [];
}

/**
 * About Face -- A Token Rotator
 * Rotates tokens based on the direction the token is moved
 *
 * by Eadorin, edzillion
 */

Hooks.once("init", () => {
	registerSettings();
	game.aboutFace = new AboutFace();
	if (game.settings.get(MODULE_ID, "indicator-state") === 1) {
		Hooks.on("hoverToken", tokenHover);
		Hooks.on("highlightObjects", highlightObjects);
	}
	game.keybindings.register(MODULE_ID, "toggleTokenRotation", {
		name: "about-face.keybindings.toggleTokenRotation.name",
		hint: "about-face.keybindings.toggleTokenRotation.hint",
		onDown: () => {
			game.aboutFace.tokenRotation = !game.aboutFace.tokenRotation;
			ui.notifications.info(
				`About Face: ${game.i18n.localize(
					`about-face.keybindings.toggleTokenRotation.tooltip.${game.aboutFace.tokenRotation}`
				)}`,
				{
					console: false,
				}
			);
		},
		restricted: false,
		precedence: CONST.KEYBINDING_PRECEDENCE.NORMAL,
	});
	game.keybindings.register(MODULE_ID, "lockRotation", {
		name: "about-face.keybindings.lockRotation.name",
		hint: "about-face.keybindings.lockRotation.hint",
		onDown: () => {
			let lockRotation;
			for (let token of canvas.tokens.controlled) {
				lockRotation = !token.document.lockRotation;
				token.document.update({ lockRotation: lockRotation });
			}
			if (lockRotation !== undefined) {
				ui.notifications.info(
					`About Face: ${game.i18n.localize(`about-face.keybindings.lockRotation.tooltip.${lockRotation}`)}`,
					{ console: false }
				);
			}
		},
		restricted: true,
		precedence: CONST.KEYBINDING_PRECEDENCE.NORMAL,
	});
});
Hooks.on("canvasReady", async () => {
	if (canvas.scene?.flags?.[MODULE_ID] == null) await canvas.scene.setFlag(MODULE_ID, "sceneEnabled", true);
	if (canvas.scene?.flags?.[MODULE_ID].sceneEnabled) {
		canvas.scene.tokens.forEach((tokenDocument) => game.aboutFace.drawAboutFaceIndicator(tokenDocument.object));
	}
});
Hooks.on("preCreateToken", onPreCreateToken);
Hooks.on("preUpdateToken", onPreUpdateToken);
Hooks.on("createToken", (tokenDocument, options, userId) => {
	if (tokenDocument.object) game.aboutFace.drawAboutFaceIndicator(tokenDocument.object);
});
Hooks.on("updateToken", (tokenDocument, changes, options, userId) => {
	if (tokenDocument.object) game.aboutFace.drawAboutFaceIndicator(tokenDocument.object);
});
Hooks.on("refreshToken", (token, options) => {
	if (options.redrawEffects) game.aboutFace.drawAboutFaceIndicator(token);
});
Hooks.on("renderSceneConfig", renderSceneConfigHandler);
Hooks.on("renderSceneConfig", asyncRenderSceneConfigHandler);
Hooks.on("renderTokenConfig", renderTokenConfigHandler);
Hooks.on("renderSettingsConfig", renderSettingsConfigHandler);
//# sourceMappingURL=about-face.js.map
