class TemplateDiceMap {
	/** Unmark the KH/KL buttons if a roll is made */
	removeAdvOnRoll = true;

	/** Shows the KH/KL buttons */
	showExtraButtons = true;

	/**
	 * The formula that will be rendered on the KH/KL buttons
	 * @returns {{}}
	 *
	 * @example Making the buttons be +1/-1 for additional logic
	 * ```js
	 * return {
	 *	kh: "+1",
	 *	kl: "-1"
	 * };
	 */
	get buttonFormulas() {
		return {
			kh: "kh",
			kl: "kl"
		};
	}

	/**
	 * The dice rows that will be shown on the dice tray.
	 * @property {String} color		Optional RGB or Hex value that colors a dice's background image. If none is preset, it will be white.
	 * @property {String} img		The path to an image that will be shown on the button. If none is present, the label will be used instead.
	 * @property {String} label		The label meant to be used when the button doesn't have a proper image, like Fate Dice or multiple dice.
	 * @property {String} tooltip	Optional tooltip that will be shown instead of the key. Useful for special dice like Genesys system's.
	 * @returns {[Object]}
	 *
	 * @example
	 * ```js Dice buttons with mixed image/label
	 * return [{
	 * 	d6: { img: "icons/dice/d6black.svg" },
	 *  "4df": { label: "Fate Dice" }
	 * }];
	 * ```
	 *
	 * @example Dice buttons with just labels
	 * ```js
	 * return [{
	 * 	d6: { label: "1d6" },
	 *  "2d6": { label: "2d6" }
	 *  "3d6": { label: "3d6" }
	 * }];
	 * ```
	 *
	 * @example Dice buttons with tooltips
	 * ```js
	 * return [{
	 * 	da: { tooltip: "Proficiency" },
	 *  ds: { tooltip: "Setback" }
	 *  df: { tooltip: "Force" }
	 * }];
	 * ```
	 */
	get dice() {
		return [
			{
				d4: { img: "icons/dice/d4black.svg" },
				d6: { img: "icons/dice/d6black.svg" },
				d8: { img: "icons/dice/d8black.svg" },
				d10: { img: "icons/dice/d10black.svg" },
				d12: { img: "icons/dice/d12black.svg" },
				d20: { img: "icons/dice/d20black.svg" },
				d100: { img: "modules/dice-calculator/assets/icons/d100black.svg" },
			}
		];
	}

	/**
	 * Labels that will be shown on the Keep Highest/Lowest button if they are shown.
	 */
	get labels() {
		return {
			advantage: "DICE_TRAY.KeepHighest",
			adv: "KH",
			disadvantage: "DICE_TRAY.KeepLowest",
			dis: "KL"
		};
	}

	/**
	 * List of additional settings to be registered during the i18nInit hook.
	 */
	get settings() {
		return {};
	}

	get textarea() {
		const chatElement = ui.sidebar.popouts.chat?.element || ui.chat.element;
		return chatElement.querySelector(".chat-form textarea");
	}

	/**
	 * Logic to set display the additiona KH/KL buttons and event listeners.
	 * @param {HTMLElement} html
	 */
	applyLayout(html) {
		if (this.showExtraButtons) {
			this._createExtraButtons(html);
			this._extraButtonsLogic(html);
		}

		/** Clicking the Roll button clears and hides all orange number flags, and unmark the KH/KL keys */
		html.querySelector(".dice-tray__roll")?.addEventListener("click", async (event) => {
			event.preventDefault();
			// Taken from FoundryVTT V13's keyEvent function
			const newEvent = document.createEvent("Event");
			newEvent.initEvent("keydown", true, true);
			newEvent.keyCode = 13;
			newEvent.key = event.code = "Enter";
			this.textarea.dispatchEvent(newEvent);
		});
	}

	applyListeners(html) {
		const diceButtons = html.querySelectorAll(".dice-tray__button");
		diceButtons.forEach((button) => {
			// Avoids moving focus to the button
			button.addEventListener("pointerdown", (event) => {
				event.preventDefault();
			});
			button.addEventListener("click", (event) => {
				event.preventDefault();
				const dataset = event.currentTarget.dataset;
				CONFIG.DICETRAY.updateChatDice(dataset, "add", html);
			});

			button.addEventListener("contextmenu", (event) => {
				event.preventDefault();
				const dataset = event.currentTarget.dataset;
				CONFIG.DICETRAY.updateChatDice(dataset, "sub", html);
			});
		});

		// Handle correcting the modifier math if it's null.
		const diceTrayInput = html.querySelector(".dice-tray__input");
		diceTrayInput?.addEventListener("input", (event) => {
			let modVal = Number(event.target.value);
			modVal = Number.isNaN(modVal) ? 0 : modVal;
			event.target.value = modVal;
			CONFIG.DICETRAY.applyModifier(html);
		});
		diceTrayInput?.addEventListener("wheel", (event) => {
			const diff = event.deltaY < 0 ? 1 : -1;
			let modVal = event.currentTarget.value;
			modVal = Number.isNaN(modVal) ? 0 : Number(modVal);
			event.currentTarget.value = modVal + diff;
			CONFIG.DICETRAY.applyModifier(html);
		});

		// Handle +/- buttons near the modifier input.
		const mathButtons = html.querySelectorAll("button.dice-tray__math");
		mathButtons?.forEach((button) => {
			button.addEventListener("click", (event) => {
				event.preventDefault();
				let modVal = Number(html.querySelector('input[name="dice.tray.modifier"]').value);
				modVal = Number.isNaN(modVal) ? 0 : modVal;

				switch (event.currentTarget.dataset.formula) {
					case "+1":
						modVal += 1;
						break;
					case "-1":
						modVal -= 1;
						break;
				}

				html.querySelector('input[name="dice.tray.modifier"]').value = modVal;
				CONFIG.DICETRAY.applyModifier(html);
			});
		});
	}

	reset() {
		CONFIG.DICETRAY._resetTray(ui.chat.element);
		CONFIG.DICETRAY._resetTray(ui.sidebar.popouts.chat?.element);
		CONFIG.DICETRAY._resetTray(CONFIG.DICETRAY.popout?.element);
	}

	/**
	 * Creates the KH/KL buttons
	 * @param {HTMLElement} html
	 */
	_createExtraButtons(html) {
		const { kh, kl } = this.buttonFormulas;
		const math = html.querySelector("#dice-tray-math");
		math.removeAttribute("hidden");
		const div = document.createElement("div");
		div.classList.add("dice-tray__stacked", "flexcol");

		const buttonAdv = document.createElement("button");
		buttonAdv.classList.add("dice-tray__ad", "dice-tray__advantage");
		buttonAdv.setAttribute("data-formula", kh);
		buttonAdv.setAttribute("data-tooltip", game.i18n.localize(this.labels.advantage));
		buttonAdv.setAttribute("data-tooltip-direction", "UP");
		buttonAdv.textContent = game.i18n.localize(this.labels.adv);

		const buttonDis = document.createElement("button");
		buttonDis.classList.add("dice-tray__ad", "dice-tray__disadvantage");
		buttonDis.setAttribute("data-formula", kl);
		buttonDis.setAttribute("data-tooltip", game.i18n.localize(this.labels.disadvantage));
		buttonDis.setAttribute("data-tooltip-direction", "UP");
		buttonDis.textContent = game.i18n.localize(this.labels.dis);

		div.appendChild(buttonAdv);
		div.appendChild(buttonDis);

		math.append(div);
	}

	/**
	 * Sets the logic for using the KH/KL buttons.
	 * This version appends KH/KL to rolls. Check DCC or SWADE for other uses.
	 * @param {HTMLElement} html
	 */
	_extraButtonsLogic(html) {
		const buttons = html.querySelectorAll(".dice-tray__ad");
		for (const button of buttons) {
			button.addEventListener("click", (event) => {
				event.preventDefault();
				const dataset = event.currentTarget.dataset;
				const chat = this.textarea;
				let chatVal = String(chat.value);
				const matchString = /\d*d\d+[khl]*/;

				// If there's a d20, toggle the current if needed.
				if (matchString.test(chatVal)) {
					// If there was previously a kh or kl, update it.
					if (/d\d+k[hl]/g.test(chatVal)) {
						chatVal = chatVal.replace(/(\d*)(d\d+)(k[hl]\d*)/g, (match, p1, p2, p3, offset, string) => {
							let diceKeep = this.updateDiceKeep(p1, p2, p3, -1, dataset.formula);
							html.querySelector(`.dice-tray__flag--${p2}`).textContent = diceKeep.count;
							return diceKeep.content;
						});
					}
					// Otherwise, add it.
					else {
						chatVal = chatVal.replace(/(\d*)(d\d+)/g, (match, p1, p2, offset, string) => {
							let diceKeep = this.updateDiceKeep(p1, p2, "", 1, dataset.formula);
							html.querySelector(`.dice-tray__flag--${p2}`).textContent = diceKeep.count;
							return diceKeep.content;
						});
					}
				}
				// else {
				// 	let diceKeep = this.updateDiceKeep("1", "d20", "", 1, dataset.formula);
				// 	html.find(".dice-tray__flag--d20").text(diceKeep.count);
				// 	chatVal += diceKeep.content;
				// }

				// Handle toggle classes.
				const toggleClass = (selector, condition) => {
					html.querySelector(selector)?.classList.toggle("active", condition);
				};
				toggleClass(".dice-tray__advantage", chatVal.includes("kh"));
				toggleClass(".dice-tray__disadvantage", chatVal.includes("kl"));
				// Update the value.
				chat.value = chatVal;
			});
		}
	}

	_resetTray(html) {
		if (!html) return;
		html.querySelector(".dice-tray__input").value = 0;
		for (const flag of html.querySelectorAll(".dice-tray__flag")) {
			flag.textContent = "";
			flag.classList.add("hide");
		}
		if (this.removeAdvOnRoll) {
			html.querySelector(".dice-tray__ad").classList.remove("active");
		}
	}

	/**
	 * Logic to apply the number on the -/+ selector.
	 * @param {HTMLElement} html
	 */
	applyModifier(html) {
		const modInput = html.querySelector(".dice-tray__input");
		if (!modInput) return;
		const modVal = Number(modInput.value);

		if (modInput.length === 0 || isNaN(modVal)) return;

		let modString = "";
		if (modVal > 0) {
			modString = `+${modVal}`;
		} else if (modVal < 0) {
			modString = `${modVal}`;
		}

		const chat = this.textarea;
		const chatVal = String(chat.value);

		const matchString = /(\+|-)(\d+)$/;
		if (matchString.test(chatVal)) {
			chat.value = chatVal.replace(matchString, modString);
		} else if (chatVal !== "") {
			chat.value = chatVal + modString;
		} else {
			const rollPrefix = this._getRollMode(html);
			chat.value = `${rollPrefix} ${modString}`;
		}
		if (/(\/r|\/gmr|\/br|\/sr) $/g.test(chat.value)) {
			chat.value = "";
		}
		this.textarea.focus();
	}

	/**
	 * Returns a string with the number of dice to be rolled.
	 * Generally simple, unless the system demands some complex use.
	 * Consider checking SWADE's implementation.
	 * @param {String} qty
	 * @param {String} dice
	 * @param {HTMLElement} html
	 * @returns {String}
	 */
	rawFormula(qty, dice, html) {
		return `${qty === "" ? 1 : qty}${dice}`;
	}

	/**
	 * Handles clicks on the dice buttons.
	 * @param {Object} dataset
	 * @param {String} direction
	 * @param {HTMLElement} html
	 * @returns
	 */
	updateChatDice(dataset, direction, html) {
		const chat = this.textarea;
		let currFormula = String(chat.value);
		if (direction === "sub" && currFormula === "") return;
		let rollPrefix = this._getRollMode(html);
		let qty = 0;
		let dice = "";

		let matchDice = dataset.formula;
		if (dataset.formula === "d10") {
			// Filter out d100s
			matchDice = "d10(?!0)";
		} else if (/^(\d+)(d.+)/.test(dataset.formula)) {
			const match = dataset.formula.match(/^(\d+)(d.+)/);
			qty = Number(match[1]);
			matchDice = match[2];
			dice = match[2];
		}
		// Catch KH/KL
		matchDice = `${matchDice}[khl]*`;

		const matchString = new RegExp(`${this.rawFormula("(\\d*)", `(${matchDice})`, html)}(?=\\+|\\-|$)`);
		if (matchString.test(currFormula)) {
			const match = currFormula.match(matchString);
			const parts = {
				qty: match.groups?.qty ?? (match[1] || "1"),
				die: match.groups?.dice ?? (match[2] || ""),
			};

			if (parts.die === "" && match[3]) {
				parts.die = match[3];
			}

			qty = direction === "add" ? Number(parts.qty) + (qty || 1) : Number(parts.qty) - (qty || 1);

			// Update the dice quantity.
			qty = qty < 1 ? "" : qty;

			if (qty === "" && direction === "sub") {
				const newMatchString = new RegExp(`${this.rawFormula("(\\d*)", `(${matchDice})`, html)}(?=\\+|\\-|$)`);
				currFormula = currFormula.replace(newMatchString, "");
				if (new RegExp(`${rollPrefix}\\s+(?!.*d\\d+.*)`).test(currFormula)) {
					currFormula = "";
				}
			} else {
				const newFormula = this.rawFormula(qty, parts.die, html);
				currFormula = currFormula.replace(matchString, newFormula);
			}
			chat.value = currFormula;
		} else {
			if (!qty) {
				qty = 1;
			}
			if (currFormula === "") {
				chat.value = `${rollPrefix} ${this.rawFormula(qty, dice || dataset.formula, html)}`;
			} else {
				const signal = (/(\/r|\/gmr|\/br|\/sr) (?!-)/g.test(currFormula)) ? "+" : "";
				currFormula = currFormula.replace(/(\/r|\/gmr|\/br|\/sr) /g, `${rollPrefix} ${this.rawFormula(qty, dice || dataset.formula, html)}${signal}`);
				chat.value = currFormula;
			}
		}

		// TODO consider separate this into another method to make overriding simpler
		// TODO e.g. cases where a button adds 2+ dice

		// Add a flag indicator on the dice.
		qty = Number(qty);
		let flagButton = [html.querySelector(`.dice-tray__flag--${dataset.formula}`)];
		if (this.popout?.rendered) {
			flagButton.push(this.popout.element.querySelector(`.dice-tray__flag--${dataset.formula}`));
		}
		if (!qty) {
			qty = direction === "add" ? 1 : 0;
		}

		for (const button of flagButton) {
			if (qty > 0) {
				button.textContent = qty;
				button.classList.remove("hide");
			} else if (qty < 0) {
				button.textContent = qty;
			} else {
				button.textContent = "";
				button.classList.add("hide");
			}
		}

		currFormula = chat.value;
		currFormula = currFormula.replace(/(\/r|\/gmr|\/br|\/sr)(( \+)| )/g, `${rollPrefix} `).replace(/\+{2}/g, "+").replace(/-{2}/g, "-").replace(/\+$/g, "");
		chat.value = currFormula;
		this.applyModifier(html);
	}

	/**
	 * Gets the selected roll mode
	 * @param {HTMLElement} html
	 * @returns {String}
	 */
	_getRollMode(html) {
		const rollMode = game.settings.get("core", "rollMode");
		switch (rollMode) {
			case "gmroll":
				return "/gmr";
			case "blindroll":
				return "/br";
			case "selfroll":
				return "/sr";
			case "publicroll":
			default:
				return "/r";
		}
	}

	/**
     * Process a formula to apply advantage or disadvantage. Should be used
     * within a regex replacer function's callback.
     *
     * @param {string} count | String match for the current dice count.
     * @param {string} dice | String match for the dice type (d20).
     * @param {string} khl | String match for kh|l (includes kh|l count).
     * @param {number} countDiff | Integer to adjust the dice count by.
     * @param {string} newKhl | Formula on the button (kh or kl).
     * @returns {object} Object with content and count keys.
     */
	updateDiceKeep(count, dice, khl, countDiff, newKhl) {
		// Start by getting the current number of dice (minimum 1).
		let keep = Number.isNumeric(count) ? Number(count) : 1;
		if (keep === 0) {
			keep = 1;
		}

		// Apply the count diff to adjust how many dice we need for adv/dis.
		let newCount = keep + countDiff;
		let newKeep = newCount - 1;

		// Handling toggling on/off advantage.
		if (khl) {
			// If switching from adv to dis or vice versa, adjust the formula to
			// simply replace the kh and kl while leaving the rest as it was prior
			// to applying the count diff.
			if (!khl.includes(newKhl)) {
				newCount = keep;
				newKeep = newCount - 1;
				khl = newKhl;
			}
			// If the adv/dis buttons were clicked after they were previously
			// applied, we need to remove them. If it's currently 2d20kh or kl,
			// change it to 1d20. Otherwise, only strip the kh or kl.
			else {
				newCount = keep > 2 ? keep : newCount;
				newKeep = 0;
			}
		}
		// If adv/dis weren't enabled, then that means we need to enable them.
		else {
			khl = newKhl;
		}

		// Limit the count to 2 when adding adv/dis to avoid accidental super advantage.
		if (newCount > 2 && newKeep > 0) {
			newCount = 2;
			newKeep = newCount - 1;
		}

		// Create the updated text string.
		let result = `${newCount > 0 ? newCount : 1}${dice}`;
		// Append kh or kl if needed.
		if (newCount > 1 && newKeep > 0) {
			result = `${result}${newKhl.includes("kh") ? "kh" : "kl"}`;
		}

		// TODO: This allows for keeping multiple dice, but in this case, we only need to keep one.
		// if (newCount > 1 && newKeep > 1) result = `${result}${newKeep}`;

		// Return an object with the updated text and the new count.
		return {
			content: result,
			count: newCount
		};
	}
}

class dccDiceMap extends TemplateDiceMap {
	// Redundant, buttons don't keep lit up on use
	removeAdvOnRoll = false;

	get buttonFormulas() {
		return {
			kh: "+1",
			kl: "-1"
		};
	}

	get dice() {
		return [
			{
				d3: { img: "modules/dice-calculator/assets/icons/d3black.svg" },
				d4: { img: "icons/dice/d4black.svg" },
				d5: { img: "modules/dice-calculator/assets/icons/d5black.svg" },
				d6: { img: "icons/dice/d6black.svg" },
				d7: { img: "modules/dice-calculator/assets/icons/d7black.svg" },
				d8: { img: "icons/dice/d8black.svg" },
				d10: { img: "icons/dice/d10black.svg" },
			},
			{
				d12: { img: "icons/dice/d10black.svg" },
				d14: { img: "modules/dice-calculator/assets/icons/d14black.svg" },
				d16: { img: "modules/dice-calculator/assets/icons/d16black.svg" },
				d20: { img: "icons/dice/d20black.svg" },
				d24: { img: "modules/dice-calculator/assets/icons/d24black.svg" },
				d30: { img: "modules/dice-calculator/assets/icons/d30black.svg" },
				d100: { img: "modules/dice-calculator/assets/icons/d100black.svg" },
			},
		];
	}

	get labels() {
		return {
			advantage: "DICE_TRAY.PlusOneDieLong",
			adv: "DICE_TRAY.PlusOneDie",
			disadvantage: "DICE_TRAY.MinusOneDieLong",
			dis: "DICE_TRAY.MinusOneDie"
		};
	}

	_extraButtonsLogic(html) {
		const buttons = html.querySelectorAll(".dice-tray__ad");
		for (const button of buttons) {
			button.addEventListener("click", (event) => {
				event.preventDefault();
				const chat = this.textarea;
				let chatVal = chat.value;
				const matchString = /(\d+)d(\d+)/;

				// Find the first dice on the line to update
				const match = chatVal.match(matchString);
				if (match) {
					const diceChain = [3, 4, 5, 6, 7, 8, 10, 12, 14, 16, 20, 24, 30];
					// Locate this die in the dice chain
					const chainIndex = diceChain.indexOf(parseInt(match[2]));
					if (chainIndex >= 0) {
						const newIndex = chainIndex + parseInt(event.currentTarget.dataset.formula);
						// Is the new index still in range?
						if (newIndex >= 0 && newIndex < diceChain.length) {
							chatVal = chatVal.replace(matchString, `${match[1]}d${diceChain[newIndex]}`);
						}
					}
				}

				// Update the value.
				chat.value = chatVal;
			});
		}
	}
}

class demonlordDiceMap extends TemplateDiceMap {

	get dice() {
		const dice = [
			{
				d3: { img: "modules/dice-calculator/assets/icons/d3black.svg" },
				d6: { img: "icons/dice/d6black.svg" },
				d20: { img: "icons/dice/d20black.svg" }
			}
		];
		return dice;
	}

	get buttonFormulas() {
		return {
			kh: "+",
			kl: "-"
		};
	}

	get labels() {
		return {
			advantage: game.i18n.localize("DL.DialogBoon"),
			adv: game.i18n.localize("DL.DialogBoon"),
			disadvantage: game.i18n.localize("DL.DialogBane"),
			dis: game.i18n.localize("DL.DialogBane")
		};
	}

	_extraButtonsLogic(html) {
		const buttons = html.querySelectorAll(".dice-tray__ad");
		for (const button of buttons) {
			button.addEventListener("click", (event) => {
				event.preventDefault();
				let sign = event.currentTarget.dataset.formula;
				const chat = this.textarea; // Assuming `this.textarea` refers to a valid element
				let chatVal = String(chat.value);
				const matchString = /(?<sign>[+-])(?<qty>\d*)d6kh/;
				const match = chatVal.match(matchString);

				if (match) {
					let qty = parseInt(match.groups.qty) || 1;
					let replacement = "";

					if (match.groups.sign === sign) {
						qty += 1;
						replacement = `${sign}${qty}d6kh`;
					} else if (qty !== 1) {
						qty -= 1;
						sign = sign === "+" ? "-" : "+";
						replacement = `${sign}${qty}d6kh`;
					}

					chatVal = chatVal.replace(matchString, replacement);
				} else {
					chatVal = `${chatVal}${sign}1d6kh`;
				}

				chat.value = chatVal;
			});
		}
	}
}

class dnd5eDiceMap extends TemplateDiceMap {
	get labels() {
		return {
			advantage: "DICE_TRAY.Advantage",
			adv: "DICE_TRAY.Adv",
			disadvantage: "DICE_TRAY.Disadvantage",
			dis: "DICE_TRAY.Dis"
		};
	}
}

class FateDiceMap extends TemplateDiceMap {
	get dice() {
		return [
			{
				d6: { img: "icons/dice/d6black.svg" },
				"4df": { label: game.i18n.localize("DICE_TRAY.FateDice")},
			}
		];
	}
}

class pf2eDiceMap extends TemplateDiceMap {
	get labels() {
		return {
			advantage: "DICE_TRAY.Fortune",
			adv: "DICE_TRAY.For",
			disadvantage: "DICE_TRAY.Misfortune",
			dis: "DICE_TRAY.Mis"
		};
	}
}

class starwarsffgDiceMap extends TemplateDiceMap {
	showExtraButtons = false;

	get dice() {
		return [
			{
				dp: {
					tooltip: game.i18n.localize("SWFFG.DiceProficiency"),
					img: "systems/starwarsffg/images/dice/starwars/yellow.png",
					color: "#fef135"
				},
				da: {
					tooltip: game.i18n.localize("SWFFG.DiceAbility"),
					img: "systems/starwarsffg/images/dice/starwars/green.png",
					color: "#46ac4f"
				},
				dc: {
					tooltip: game.i18n.localize("SWFFG.DiceChallenge"),
					img: "systems/starwarsffg/images/dice/starwars/red.png",
					color: "#751317"
				},
				di: {
					tooltip: game.i18n.localize("SWFFG.DiceDifficulty"),
					img: "systems/starwarsffg/images/dice/starwars/purple.png",
					color: "#52287e"
				},
				db: {
					tooltip: game.i18n.localize("SWFFG.DiceBoost"),
					img: "systems/starwarsffg/images/dice/starwars/blue.png",
					color: "#76c3db"
				},
				ds: {
					tooltip: game.i18n.localize("SWFFG.DiceSetback"),
					img: "systems/starwarsffg/images/dice/starwars/black.png",
					color: "#141414"
				},
				df: {
					tooltip: game.i18n.localize("SWFFG.DiceForce"),
					img: "systems/starwarsffg/images/dice/starwars/whiteHex.png",
					color: "#ffffff"
				}
			}
		];
	}
}

class SWADEDiceMap extends TemplateDiceMap {
	removeAdvOnRoll = false;

	get dice() {
		const dice = [
			{
				d4: { img: "icons/dice/d4black.svg" },
				d6: { img: "icons/dice/d6black.svg" },
				d8: { img: "icons/dice/d8black.svg" },
				d10: { img: "icons/dice/d10black.svg" },
				d12: { img: "icons/dice/d12black.svg" }
			}
		];
		return dice;
	}

	get labels() {
		return {
			advantage: game.i18n.localize("DICE_TRAY.WildDie"),
			adv: game.i18n.localize("DICE_TRAY.Wild"),
			disadvantage: game.i18n.localize("DICE_TRAY.ExplodingDie"),
			dis: game.i18n.localize("DICE_TRAY.Ace")
		};
	}

	get settings() {
		return {
			wildDieBehavior: {
				name: "DICE_TRAY.SETTINGS.SWADE.wildDieBehavior.name",
				hint: "DICE_TRAY.SETTINGS.SWADE.wildDieBehavior.hint",
				default: false,
				type: Boolean
			}
		};
	}

	_extraButtonsLogic(html) {
		const advantage = html.querySelector(".dice-tray__advantage");
		const disadvantage = html.querySelector(".dice-tray__disadvantage");
		advantage.addEventListener("click", (event) => {
			event.preventDefault();
			advantage.classList.toggle("active");
			if (!advantage.classList.has("active")) {
				disadvantage.classList.add("active");
			}
		});
		disadvantage.addEventListener("click", (event) => {
			event.preventDefault();
			disadvantage.classList.toggle("active");
			if (disadvantage.classList.has("active")) {
				advantage.classList.remove("active");
			}
		});
	}

	rawFormula(qty, dice, html) {
		let roll_suffix = "";
		let add_wild = html.querySelector(".dice-tray__advantage").classList.has("active");

		if (html.querySelector(".dice-tray__disadvantage").classList.has("active")) {
			roll_suffix = "x=";
		}

		if (add_wild) {
			if (!game.settings.get("dice-calculator", "wildDieBehavior")) {
				return `{${qty === "" ? 1 : qty}${dice}${roll_suffix},1d6${roll_suffix}}kh`;
			}

			dice = dice.replace("(", "").replace(")", "");
			if (!Number.isNumeric(qty)) {
				return `{1(?<dice>${dice})${roll_suffix}.*,1d6${roll_suffix}}kh(?<qty>\\d*)`;
			}
			return `{${`1${dice}${roll_suffix},`.repeat(qty)}1d6${roll_suffix}}kh${qty}`;
		}
		return `${qty === "" ? 1 : qty}${dice}${roll_suffix}`;
	}
}

class HeXXen1733DiceMap extends TemplateDiceMap {
	/** Shows the KH/KL buttons */
	showExtraButtons = false;

	get dice() {
		return [
			{
				h: {
					tooltip: "HeXXenwürfel",
					img: "systems/hexxen-1733/img/dice/svg/erfolgswuerfel_einfach.svg",
					color: "#00a806"
				},
				s: {
					tooltip: "Segnungswürfel",
					img: "systems/hexxen-1733/img/dice/svg/erfolgswuerfel_doppel.svg",
					color: "#d1c5a8"
				},
				b: {
					tooltip: "Blutwürfel",
					img: "systems/hexxen-1733/img/dice/svg/blutwuerfel_3.svg",
					color: "#a74937"
				},
				e: {
					tooltip: "Elixierwürfel",
					img: "systems/hexxen-1733/img/dice/svg/elixirwuerfel_5.svg",
					color: "#4c7ba0"
				}
			}
		];
	}

	applyModifier(html) {
		const modInput = html.querySelector(".dice-tray__input");
		if (!modInput) return;
		const modVal = Number(modInput.value);

		if (modInput.length === 0 || isNaN(modVal)) return;

		let modString = "";
		let modTemp = "";
		if (modVal > 0) {
			modString = `${modVal}+`;
		} else if (modVal < 0) {
			modTemp = Math.abs(modVal);
			modString = `${modTemp}-`;
		}

		const chat = this.textarea;
		const chatVal = String(chat.value);

		const matchString = /(\d+)(\+|-)$/;
		if (matchString.test(chatVal)) {
			chat.value = chatVal.replace(matchString, modString);
		} else if (chatVal !== "") {
			chat.value = chatVal + modString;
		} else {
			const rollPrefix = this._getRollMode(html);
			chat.value = `${rollPrefix} ${modString}`;
		}

		if (/(\/r|\/gmr|\/br|\/sr) $/g.test(chat.value)) {
			chat.value = "";
		}
	}

	updateChatDice(dataset, direction, html) {
		const chat = this.textarea;
		let currFormula = String(chat.value);
		if (direction === "sub" && currFormula === "") return;
		let newFormula = null;
		let rollPrefix = this._getRollMode(html);
		let qty = 0;

		let match_dice = dataset.formula;
		const matchString = new RegExp(`${this.rawFormula("(\\d*)", `(${match_dice})`, html)}(?=[0-9]|$)`);

		if (matchString.test(currFormula)) {
			const match = currFormula.match(matchString);
			const parts = {
				txt: match[0] || "",
				qty: match[1] || "1",
				die: match[2] || "",
			};

			if (parts.die === "" && match[3]) {
				parts.die = match[3];
			}

			qty = direction === "add" ? Number(parts.qty) + (qty || 1) : Number(parts.qty) - (qty || 1);

			// Update the dice quantity.
			qty = qty < 1 ? "" : qty;

			if (qty === "" && direction === "sub") {
				newFormula = "";
				let regexxx =`${this.rawFormula("(\\d+)", `(${match_dice})`, html)}(?=[0-9]|$)`;
				const newMatchString = new RegExp(regexxx);
				currFormula = currFormula.replace(newMatchString, newFormula);
				if (!(/(\d+[hsbe+-])/.test(currFormula))) {

					currFormula = "";
				}
			} else {
				newFormula = this.rawFormula(qty, parts.die, html);
				currFormula = currFormula.replace(matchString, newFormula);
			}
			chat.value = currFormula;
		} else {
			if (!qty) {
				qty = 1;
			}
			if (currFormula === "") {
				chat.value = `${rollPrefix} ${this.rawFormula(qty, dataset.formula, html)}`;
			} else {
				const signal = (/(\/r|\/gmr|\/br|\/sr) (?!-)/g.test(currFormula)) ? "" : "";
				currFormula = currFormula.replace(/(\/r|\/gmr|\/br|\/sr) /g, `${rollPrefix} ${this.rawFormula(qty, dataset.formula, html)}${signal}`);
				chat.value = currFormula;
			}
		}
		// TODO consider separate this into another method to make overriding simpler
		// TODO e.g. cases where a button adds 2+ dice

		// Add a flag indicator on the dice.
		qty = Number(qty);
		const flagButton = html.querySelector(`.dice-tray__flag--${dataset.formula}`);
		if (!qty) {
			qty = direction === "add" ? 1 : 0;
		}

		if (qty > 0) {
			flagButton.textContent = qty;
			flagButton.classList.remove("hide");
		} else if (qty < 0) {
			flagButton.textContent = qty;
		} else {
			flagButton.textContent = "";
			flagButton.classList.add("hide");
		}

		currFormula = chat.value;
		currFormula = currFormula.replace(/(\/r|\/gmr|\/br|\/sr)(( \+)| )/g, `${rollPrefix} `)
			.replace(/\+{2}/g, "+")
			.replace(/-{2}/g, "-");
		chat.value = currFormula;
		this.applyModifier(html);
	}
}

class GrimwildDiceMap extends TemplateDiceMap {
	// Ironically, this is used to *remove* extra buttons like the input.
	// @todo update the parent class to add something like a render hook
	// for a more accurate place to modify the final markup.
	showExtraButtons = true;

	// Prepare dice buttons.
	get dice() {
		return [
			{
				d: {
					// img: "icons/dice/d6black.svg",
					tooltip: "Dice",
					label: "<i class=\"fas fa-dice-d6\"></i> d",
					direction: "LEFT"
				},
				t: {
					// img: "icons/dice/d8black.svg",
					tooltip: "Thorns",
					label: "<i class=\"fas fa-dice-d8\"></i> t",
					direction: "LEFT"
				},
				p: {
					// img: "icons/dice/d6black.svg",
					tooltip: "Pool",
					label: "<i class=\"fas fa-dice-d6\"></i> Pool",
					direction: "LEFT"
				},
			}
		];
	}

	// Override the chat formula logic.
	updateChatDice(dataset, direction, html) {
		// Get the DOM element rather than the jQuery object.
		html = html[0];
		// Retrieve the current chat value.
		const chat = html.querySelector("#chat-form textarea");
		let currFormula = String(chat.value);
		// Exit early if there's nothing in chat and this is a remove operation.
		if (direction === "sub" && currFormula === "") return;
		// Grab the dice roll mode from chat.
		let rollPrefix = this._getRollMode(html);
		// Store the current dice and thorn values for later.
		let dice = "";
		let thorns = "";

		// If the current formula is empty, set it to the roll prefix as our baseline.
		if (currFormula === "") currFormula = rollPrefix;

		// Prepare a string of possible roll types for the regex. This should also
		// catch any manually written roll types, like "/gmroll".
		const rollModes = [
			"/roll", "/r",
			"/publicroll", "/pr",
			"/gmroll", "/gmr",
			"/blindroll", "/broll", "/br",
			"/selfroll", "/sr"
		].join("|");

		// Convert our operation into math.
		let delta = direction === "add" ? 1 : -1;

		/**
		 * Regex for the dice expression. Examples: /r 4d2t, /gmr 2d, /br 4p
		 * Parts:
		 * (${rollModes})+ - Will be the /r, /gmroll, etc.
		 * \\s* - Whitespace between roll and formula.
		 * (\\d+[dp])* - Dice or pool, 4d, 4p, etc.
		 * (\\d+t)* - Thorns, 4t
		 * (.)* - Catch all for trailing characters. Used to snip off extras like "/r 4d6" becoming "/r 4d"
		 */
		const rollTextRegex = new RegExp(`(${rollModes})+\\s*(\\d+[dp])*(\\d+t)*(.)*`);
		// Run the regex with capture groups for targeted replacement.
		currFormula = currFormula.replace(rollTextRegex, (match, rollMode, diceMatch, thornsMatch, trailMatch) => {
			// If this is a remove operation and no dice were found, exit early.
			if (direction === "sub" && !diceMatch) {
				return match;
			}

			// Handle dice and pools.
			if (dataset.formula === "d" || dataset.formula === "p") {
				if (diceMatch) {
					diceMatch = diceMatch.replace(/(\d+)([dp])/, (subMatch, digit, letter) => {
						const newDigit = Number(digit) + delta;
						return newDigit > 0 ? `${newDigit}${dataset.formula}` : "";
					});

					if (!diceMatch && thornsMatch) {
						thornsMatch = "";
					}
				}
				else if (delta > 0) {
					diceMatch = `1${dataset.formula}`;
				}

				if (thornsMatch && dataset.formula === "p") {
					thornsMatch = "";
				}
			}

			// Handle thorns.
			if (dataset.formula === "t") {
				if (thornsMatch) {
					thornsMatch = thornsMatch.replace(/(\d+)(t)/, (subMatch, digit, letter) => {
						const newDigit = Number(digit) + delta;
						return newDigit > 0 ? `${newDigit}${letter}` : "";
					});
				}
				else if (delta > 0) {
					thornsMatch = "1t";
				}

				if (!diceMatch) {
					diceMatch = "1d";
				}

				diceMatch = diceMatch.replace("p", "d");
			}

			// Update variables.
			dice = diceMatch;
			thorns = thornsMatch;

			// Update the chat string.
			return `${rollPrefix} ${diceMatch}${thornsMatch ?? ""}`;
		});

		// Update flags over dice buttons. Use document instead of html so that we
		// also catch the popout element if present.
		let flagButton = document.querySelectorAll(".dice-tray__flag");
		flagButton.forEach((button) => {
			const buttonType = button.closest("button")?.dataset?.formula ?? false;
			// Update dice button.
			if (buttonType === "d") {
				if (dice && ["d", "t"].includes(dataset.formula)) {
					button.textContent = dice;
					button.classList.remove("hide");
				}
				else {
					button.textContent = "";
					button.classList.add("hide");
				}
			}
			// Update thorn button.
			else if (buttonType === "t") {
				if (thorns && ["d", "t"].includes(dataset.formula)) {
					button.textContent = thorns;
					button.classList.remove("hide");
				}
				else {
					button.textContent = "";
					button.classList.add("hide");
				}
			}
			// Update pool button.
			else if (buttonType === "p") {
				if (dice && dataset.formula === "p") {
					button.textContent = dice;
					button.classList.remove("hide");
				}
				else {
					button.textContent = "";
					button.classList.add("hide");
				}
			}
		});

		// Update chat area if the formula is valid.
		if (rollTextRegex.test(currFormula)) {
			// If there are dice, apply the formula. Otherwise, empty it.
			chat.value = dice ? currFormula : "";
		}
	}

	/**
	 * Remove buttons unused by Grimwild.
	 * @param {HTMLElement} html
	 */
	_createExtraButtons(html) {
		html = html[0];
		html.querySelector(".dice-tray__math--sub").remove();
		html.querySelector(".dice-tray__math--add").remove();
		html.querySelector(".dice-tray__input").remove();
	}
}

var keymaps = /*#__PURE__*/Object.freeze({
	__proto__: null,
	Template: TemplateDiceMap,
	dcc: dccDiceMap,
	demonlord: demonlordDiceMap,
	dnd5e: dnd5eDiceMap,
	ModularFate: FateDiceMap,
	fateCoreOfficial: FateDiceMap,
	fatex: FateDiceMap,
	pf2e: pf2eDiceMap,
	starwarsffg: starwarsffgDiceMap,
	swade: SWADEDiceMap,
	hexxen1733: HeXXen1733DiceMap,
	grimwild: GrimwildDiceMap
});

const { ApplicationV2, HandlebarsApplicationMixin } = foundry.applications.api;

class DiceTrayPopOut extends HandlebarsApplicationMixin(ApplicationV2) {
	static DEFAULT_OPTIONS = {
		id: "dice-tray-popout",
		tag: "aside",
		position: {
			width: ui?.sidebar?.options.width ?? 300
		},
		window: {
			title: "DICE_TRAY.DiceTray",
			icon: "fas fa-dice-d20",
			minimizable: true
		}
	};

	static PARTS = {
		list: {
			id: "list",
			template: "modules/dice-calculator/templates/tray.html",
		}
	};

	async _renderFrame(options) {
		const frame = await super._renderFrame(options);
		this.window.close.remove(); // Prevent closing
		return frame;
	}

	async close(options={}) {
		if ( !options.closeKey ) return super.close(options);
		return this;
	}

	get chatElement() {
		return ui.sidebar.popouts.chat?.element || ui.chat.element;
	}

	async _onFirstRender(context, options) {
		await super._onFirstRender(context, options);
		const position = game.settings.get("dice-calculator", "popoutPosition");
		const left = position.left ?? ui.nav?.element.getBoundingClientRect().left;
		const top = position.top ?? ui.controls?.element.getBoundingClientRect().top;
		options.position = {...options.position, left, top};
	}

	_onRender(context, options) {
		super._onRender(context, options);
		CONFIG.DICETRAY.applyListeners(this.element);
		CONFIG.DICETRAY.applyLayout(this.element);
	}

	async _prepareContext(_options) {
		return {
			dicerows: game.settings.get("dice-calculator", "diceRows"),
		};
	}

	setPosition(position) {
		const superPosition = super.setPosition(position);
		const { left, top } = superPosition;
		game.settings.set("dice-calculator", "popoutPosition", { left, top });
		return superPosition;
	}
}

class DiceCreator extends FormApplication {
	constructor(object, options = {}) {
		super(object, options);
		Hooks.once("closeDiceRowSettings", () => this.close());
	}

	static get defaultOptions() {
		return foundry.utils.mergeObject(super.defaultOptions, {
			id: "dice-creator-form",
			title: "DICE_TRAY.SETTINGS.DiceCreator",
			template: "./modules/dice-calculator/templates/DiceCreator.hbs",
			classes: ["sheet", "dice-tray-dice-creator"],
			width: 400,
			closeOnSubmit: true,
		});
	}

	getData(options) {
		const { dice, settings } = this.object;
		const nextRow = this.object.diceRows.findIndex((row) => Object.keys(row).length < 7);
		const rowIndex = Math.clamp(nextRow, 0, this.object.diceRows.length) + 1;
		return {
			dice,
			diceRows: this.object.diceRows, // this.diceRows,
			value: dice?.row ?? rowIndex,
			nextRow: rowIndex,
			maxRows: rowIndex + 1,
			settings
		};
	}

	async activateListeners(html) {
		super.activateListeners(html);
		html.find("button[name=cancel]").on("click", async (event) => {
			this.close();
		});
	}

	async _updateObject(event, formData) {
		let { dice, row } = foundry.utils.expandObject(formData);
		row -= 1;
		if (this.object.dice && dice.row !== row) {
			const key = this.object.dice.originalKey;
			delete this.object.form.diceRows[row][key];
		}
		if (row + 1 > this.object.form.diceRows.length) {
			this.object.form.diceRows.push({});
		}
		const cleanKey = Object.fromEntries(Object.entries(dice).filter(([k, v]) => k !== "key" && v !== ""));
		if (!cleanKey.img && !cleanKey.label) {
			cleanKey.label = dice.key;
		}
		this.object.form.diceRows[row][dice.key] = cleanKey;
		this.object.form.render(true);
	}
}

class DiceRowSettings extends FormApplication {
	constructor(object, options = {}) {
		super(object, options);
		this.diceRows = foundry.utils.deepClone(game.settings.get("dice-calculator", "diceRows"));
	}

	static get defaultOptions() {
		return foundry.utils.mergeObject(super.defaultOptions, {
			id: "dice-row-form",
			title: "DICE_TRAY.SETTINGS.DiceRowSettings",
			template: "./modules/dice-calculator/templates/DiceRowSettings.hbs",
			classes: ["sheet", "dice-tray-row-settings"],
			width: 320,
			height: "auto",
			closeOnSubmit: true,
		});
	}

	settings;

	static settingsKeys = ["compactMode", "hideNumberInput", "hideRollButton"];

	getData(options) {
		this.settings ??= DiceRowSettings.settingsKeys.reduce((obj, key) => {
			obj[key] = game.settings.get("dice-calculator", key);
			return obj;
		}, {});
		return {
			diceRows: this.diceRows,
			settings: this.settings
		};
	}

	async activateListeners(html) {
		super.activateListeners(html);
		CONFIG.DICETRAY.applyLayout(html[0]);
		html.find("input").on("click", async (event) => {
			const { checked, name } = event.currentTarget;
			this.settings[name] = checked;
			this.render(true);
		});
		html.find(".dice-tray button").on("click", async (event) => {
			event.preventDefault();
		});
		html.find(".dice-tray__button").on("click", async (event) => {
			event.preventDefault();
			const { formula: key, tooltip } = Object.keys(event.target.parentElement.dataset).length
				? event.target.parentElement.dataset
				: event.target.dataset;
			const row = this.diceRows.findIndex((r) => r[key]);
			const diceData = this.diceRows[row][key];
			const { color, img, label } = diceData;
			new DiceCreator({
				form: this,
				diceRows: this.diceRows,
				dice: {
					key,
					originalKey: key, // In case the key is changed later.
					color,
					img,
					label,
					tooltip: tooltip !== key ? tooltip : "",
					row: row + 1,
				},
				settings: this.settings
			}).render(true);
		});
		html.find(".dice-tray__button").on("contextmenu", async (event) => {
			event.preventDefault();
			const { formula: key } = Object.keys(event.target.parentElement.dataset).length
				? event.target.parentElement.dataset
				: event.target.dataset;
			const row = this.diceRows.findIndex((r) => r[key]);
			delete this.diceRows[row][key];
			if (!Object.keys(this.diceRows[row]).length) {
				this.diceRows.splice(1, row);
			}
			this.render(false);
		});
		html.find("button[name=add]").on("click", async (event) => {
			if (Object.keys(this.diceRows[0]).length >= 7) {
				return ui.notifications.notify("You're in Compact Mode and have too many dice. Edit or delete an existing dice before adding any more.");
			}
			new DiceCreator({
				form: this,
				diceRows: this.diceRows,
				settings: this.settings
			}).render(true);
		});
		html.find("button[name=cancel]").on("click", async (event) => {
			this.close();
		});
		html.find("button[name=reset]").on("click", async (event) => {
			this.diceRows = game.settings.settings.get("dice-calculator.diceRows").default;
			this.render(false);
		});
	}

	async _updateObject(event, formData) {
		let requiresWorldReload = false;
		await Promise.all(
			DiceRowSettings.settingsKeys.map(async (key) => {
				const current = game.settings.get("dice-calculator", key);
				if (current !== this.settings[key]) {
					await game.settings.set("dice-calculator", key, this.settings[key]);
					requiresWorldReload = true;
				}
			})
		);
		const current = game.settings.get("dice-calculator", "diceRows");
		if (JSON.stringify(this.diceRows) !== JSON.stringify(current)) {
			await game.settings.set("dice-calculator", "diceRows", this.diceRows);
		}
		if (requiresWorldReload) await SettingsConfig.reloadConfirm({ world: true });
	}
}

/**
 * Javascript imports don't support dashes so the workaround
 * for systems with dashes in their names is to create this map.
 */
const KEYS = {
	"fate-core-official": "fateCoreOfficial",
	"hexxen-1733": "hexxen1733"
};

function registerSettings() {
	game.settings.registerMenu("dice-calculator", "DiceRowSettings", {
		name: "DICE_TRAY.SETTINGS.DiceRowSettings",
		label: "DICE_TRAY.SETTINGS.DiceRowSettings",
		icon: "fas fa-cogs",
		type: DiceRowSettings,
		restricted: true,
	});

	game.settings.register("dice-calculator", "hideAdv", {
		name: game.i18n.localize("DICE_TRAY.SETTINGS.hideAdv.name"),
		hint: game.i18n.localize("DICE_TRAY.SETTINGS.hideAdv.hint"),
		scope: "world",
		config: CONFIG.DICETRAY.constructor.name === "TemplateDiceMap",
		default: false,
		type: Boolean,
		requiresReload: true
	});

	// Menu Settings

	game.settings.register("dice-calculator", "diceRows", {
		scope: "world",
		config: false,
		default: CONFIG.DICETRAY.dice,
		type: Array,
	});

	game.settings.register("dice-calculator", "compactMode", {
		scope: "world",
		config: false,
		default: false,
		type: Boolean,
	});
	game.settings.register("dice-calculator", "hideNumberInput", {
		scope: "world",
		config: false,
		default: false,
		type: Boolean,
	});
	game.settings.register("dice-calculator", "hideRollButton", {
		scope: "world",
		config: false,
		default: false,
		type: Boolean,
	});

	game.settings.register("dice-calculator", "popout", {
		name: "DICE_TRAY.SETTINGS.popout.name",
		hint: "DICE_TRAY.SETTINGS.popout.hint",
		scope: "user",
		config: true,
		default: "none",
		choices: {
			none: "",
			tokens: game.i18n.localize("CONTROLS.GroupToken"),
			all: game.i18n.localize("DICE_TRAY.SETTINGS.popout.options.all"),
		},
		type: String,
		onChange: async () => await ui.controls.render({ reset: true })
	});

	game.settings.register("dice-calculator", "autoOpenPopout", {
		name: "DICE_TRAY.SETTINGS.autoOpenPopout.name",
		hint: "DICE_TRAY.SETTINGS.autoOpenPopout.hint",
		scope: "user",
		config: true,
		default: false,
		type: Boolean
	});

	game.settings.register("dice-calculator", "popoutPosition", {
		scope: "user",
		config: false,
		default: {},
		type: Object
	});

	for (const [key, data] of Object.entries(CONFIG.DICETRAY.settings)) {
		game.settings.register("dice-calculator", key, foundry.utils.mergeObject(
			{
				scope: "world",
				config: true
			},
			data
		));
	}
}

// Initialize module
Hooks.once("init", () => {
	foundry.applications.handlebars.loadTemplates(["modules/dice-calculator/templates/tray.html"]);
});

Hooks.once("i18nInit", () => {
	const newMaps = foundry.utils.deepClone(keymaps);

	Hooks.callAll("dice-calculator.keymaps", newMaps, newMaps.Template);
	const supportedSystemMaps = Object.keys(newMaps).join("|");
	const systemMapsRegex = new RegExp(`^(${supportedSystemMaps})$`);
	const providerStringMaps = getProviderString(systemMapsRegex) || "Template";
	CONFIG.DICETRAY = new newMaps[providerStringMaps]();

	registerSettings();
	game.keybindings.register("dice-calculator", "popout", {
		name: "DICE_TRAY.KEYBINGINDS.popout.name",
		onDown: async () => {
			await togglePopout();
			if (game.settings.get("dice-calculator", "popout") === "none") return;
			const tool = ui.controls.control.tools.diceTray;
			if (tool) {
				tool.active = !tool.active;
				ui.controls.render();
			}
		}
	});
});

Hooks.once("ready", () => {
	if (game.settings.get("dice-calculator", "autoOpenPopout")) togglePopout();
});

function getProviderString(regex) {
	const id = game.system.id;
	if (id in KEYS) {
		return KEYS[id];
	} else if (regex.test(id)) {
		return id;
	}
	return "";
}

async function togglePopout() {
	CONFIG.DICETRAY.popout ??= new DiceTrayPopOut();
	if (CONFIG.DICETRAY.popout.rendered) await CONFIG.DICETRAY.popout.close({ animate: false });
	else await CONFIG.DICETRAY.popout.render(true);
}

Hooks.on("renderChatLog", async (chatlog, html, data) => {
	// Prepare the dice tray for rendering.
	let chatForm = html.querySelector(".chat-form");
	const options = {
		dicerows: game.settings.get("dice-calculator", "diceRows"),
		settings: DiceRowSettings.settingsKeys.reduce((obj, key) => {
			obj[key] = game.settings.get("dice-calculator", key);
			return obj;
		}, {})
	};

	const content = await foundry.applications.handlebars.renderTemplate("modules/dice-calculator/templates/tray.html", options);

	if (content.length > 0) {
		chatForm.insertAdjacentHTML("afterend", content);
		CONFIG.DICETRAY.applyListeners(chatForm.nextElementSibling);
		CONFIG.DICETRAY.applyLayout(html);
	}
});

Hooks.on("getSceneControlButtons", (controls) => {
	const popout = game.settings.get("dice-calculator", "popout");
	if (popout === "none") return;
	const autoOpenPopout = game.settings.get("dice-calculator", "autoOpenPopout");
	const addButton = (control) => {
		control.tools.diceTray = {
			name: "diceTray",
			title: "Dice Tray",
			icon: "fas fa-dice-d20",
			onChange: () => togglePopout(),
			active: CONFIG.DICETRAY.popout?.rendered || (!game.ready && autoOpenPopout),
			toggle: true,
		};
	};
	if (popout === "tokens") addButton(controls.tokens);
	else Object.keys(controls).forEach((c) => addButton(controls[c]));
});

Hooks.on("chatMessage", () => CONFIG.DICETRAY.reset());
//# sourceMappingURL=dice-calculator.js.map
