import ApplicationV2 from "../../api/application.mjs";
import ChatLog from "../tabs/chat.mjs";
import ChatMessage from "@client/documents/chat-message.mjs";

/**
 * @import {ApplicationConfiguration, ApplicationRenderOptions} from "../_types.mjs"
 */

/**
 * @typedef {ApplicationConfiguration} ChatPopoutConfiguration
 * @property {ChatMessage} message  The message being rendered.
 */

/**
 * A simple application for rendering a single chat message in its own frame.
 * @extends {ApplicationV2<ChatPopoutConfiguration, ApplicationRenderOptions>}
 */
export default class ChatPopout extends ApplicationV2 {
  constructor(options={}) {
    super(options);
    if ( !(options.message instanceof ChatMessage) ) throw new Error("The ChatPopout application must be provided "
      + "a ChatMessage Document to render.");
    this.#message = options.message;
  }

  /** @override */
  static DEFAULT_OPTIONS = {
    classes: ["chat-popout", "themed", "theme-light"],
    position: {
      width: 300
    }
  };

  /* -------------------------------------------- */
  /*  Properties                                  */
  /* -------------------------------------------- */

  /**
   * The message being rendered.
   * @type {ChatMessage}
   */
  get message() {
    return this.#message;
  }

  #message;

  /** @override */
  get title() {
    if ( this.message.title !== undefined ) return this.message.title;
    if ( this.message.flavor !== undefined ) {
      return new DOMParser().parseFromString(this.message.flavor, "text/html").body.textContent;
    }
    if ( this.message.speaker.alias !== undefined ) return this.message.speaker.alias;
    return "";
  }

  /* -------------------------------------------- */
  /*  Methods                                     */
  /* -------------------------------------------- */

  /** @inheritDoc */
  _initializeApplicationOptions(options) {
    const applicationOptions = super._initializeApplicationOptions(options);
    applicationOptions.uniqueId = `chat-popout-${options.message.id}`;
    return applicationOptions;
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  _onClose(options) {
    super._onClose(options);
    delete this.message.apps[this.id];
  }

  /* -------------------------------------------- */

  /** @inheritDoc */
  async _onFirstRender(context, options) {
    await super._onFirstRender(context, options);
    this.message.apps[this.id] = this;
  }

  /* -------------------------------------------- */

  /** @override */
  async _renderHTML(context, options) {
    return ChatLog.renderMessage(this.message, { canDelete: false });
  }

  /* -------------------------------------------- */

  /** @override */
  _replaceHTML(result, content, options) {
    content.replaceChildren(result);
  }
}
