'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var core = require('@pixi/core');
var display = require('@pixi/display');
var text = require('@pixi/text');
var CountLimiter = require('./CountLimiter.js');

function findMultipleBaseTextures(item, queue) {
  let result = false;
  if (item?._textures?.length) {
    for (let i = 0; i < item._textures.length; i++) {
      if (item._textures[i] instanceof core.Texture) {
        const baseTexture = item._textures[i].baseTexture;
        if (!queue.includes(baseTexture)) {
          queue.push(baseTexture);
          result = true;
        }
      }
    }
  }
  return result;
}
function findBaseTexture(item, queue) {
  if (item.baseTexture instanceof core.BaseTexture) {
    const texture = item.baseTexture;
    if (!queue.includes(texture)) {
      queue.push(texture);
    }
    return true;
  }
  return false;
}
function findTexture(item, queue) {
  if (item._texture && item._texture instanceof core.Texture) {
    const texture = item._texture.baseTexture;
    if (!queue.includes(texture)) {
      queue.push(texture);
    }
    return true;
  }
  return false;
}
function drawText(_helper, item) {
  if (item instanceof text.Text) {
    item.updateText(true);
    return true;
  }
  return false;
}
function calculateTextStyle(_helper, item) {
  if (item instanceof text.TextStyle) {
    const font = item.toFontString();
    text.TextMetrics.measureFont(font);
    return true;
  }
  return false;
}
function findText(item, queue) {
  if (item instanceof text.Text) {
    if (!queue.includes(item.style)) {
      queue.push(item.style);
    }
    if (!queue.includes(item)) {
      queue.push(item);
    }
    const texture = item._texture.baseTexture;
    if (!queue.includes(texture)) {
      queue.push(texture);
    }
    return true;
  }
  return false;
}
function findTextStyle(item, queue) {
  if (item instanceof text.TextStyle) {
    if (!queue.includes(item)) {
      queue.push(item);
    }
    return true;
  }
  return false;
}
const _BasePrepare = class {
  constructor(renderer) {
    this.limiter = new CountLimiter.CountLimiter(_BasePrepare.uploadsPerFrame);
    this.renderer = renderer;
    this.uploadHookHelper = null;
    this.queue = [];
    this.addHooks = [];
    this.uploadHooks = [];
    this.completes = [];
    this.ticking = false;
    this.delayedTick = () => {
      if (!this.queue) {
        return;
      }
      this.prepareItems();
    };
    this.registerFindHook(findText);
    this.registerFindHook(findTextStyle);
    this.registerFindHook(findMultipleBaseTextures);
    this.registerFindHook(findBaseTexture);
    this.registerFindHook(findTexture);
    this.registerUploadHook(drawText);
    this.registerUploadHook(calculateTextStyle);
  }
  upload(item) {
    return new Promise((resolve) => {
      if (item) {
        this.add(item);
      }
      if (this.queue.length) {
        this.completes.push(resolve);
        if (!this.ticking) {
          this.ticking = true;
          core.Ticker.system.addOnce(this.tick, this, core.UPDATE_PRIORITY.UTILITY);
        }
      } else {
        resolve();
      }
    });
  }
  tick() {
    setTimeout(this.delayedTick, 0);
  }
  prepareItems() {
    this.limiter.beginFrame();
    while (this.queue.length && this.limiter.allowedToUpload()) {
      const item = this.queue[0];
      let uploaded = false;
      if (item && !item._destroyed) {
        for (let i = 0, len = this.uploadHooks.length; i < len; i++) {
          if (this.uploadHooks[i](this.uploadHookHelper, item)) {
            this.queue.shift();
            uploaded = true;
            break;
          }
        }
      }
      if (!uploaded) {
        this.queue.shift();
      }
    }
    if (!this.queue.length) {
      this.ticking = false;
      const completes = this.completes.slice(0);
      this.completes.length = 0;
      for (let i = 0, len = completes.length; i < len; i++) {
        completes[i]();
      }
    } else {
      core.Ticker.system.addOnce(this.tick, this, core.UPDATE_PRIORITY.UTILITY);
    }
  }
  registerFindHook(addHook) {
    if (addHook) {
      this.addHooks.push(addHook);
    }
    return this;
  }
  registerUploadHook(uploadHook) {
    if (uploadHook) {
      this.uploadHooks.push(uploadHook);
    }
    return this;
  }
  add(item) {
    for (let i = 0, len = this.addHooks.length; i < len; i++) {
      if (this.addHooks[i](item, this.queue)) {
        break;
      }
    }
    if (item instanceof display.Container) {
      for (let i = item.children.length - 1; i >= 0; i--) {
        this.add(item.children[i]);
      }
    }
    return this;
  }
  destroy() {
    if (this.ticking) {
      core.Ticker.system.remove(this.tick, this);
    }
    this.ticking = false;
    this.addHooks = null;
    this.uploadHooks = null;
    this.renderer = null;
    this.completes = null;
    this.queue = null;
    this.limiter = null;
    this.uploadHookHelper = null;
  }
};
let BasePrepare = _BasePrepare;
BasePrepare.uploadsPerFrame = 4;

exports.BasePrepare = BasePrepare;
//# sourceMappingURL=BasePrepare.js.map
