import { Matrix, ObjectRenderer, QuadUv, State, Shader, WRAP_MODES, Color, utils, ExtensionType, extensions } from '@pixi/core';
import gl2FragmentSrc from './sprite-tiling.mjs';
import gl2VertexSrc from './sprite-tiling2.mjs';
import gl1FragmentSrc from './sprite-tiling-fallback.mjs';
import gl1VertexSrc from './sprite-tiling-fallback2.mjs';
import fragmentSimpleSrc from './sprite-tiling-simple.mjs';

const tempMat = new Matrix();
class TilingSpriteRenderer extends ObjectRenderer {
  constructor(renderer) {
    super(renderer);
    renderer.runners.contextChange.add(this);
    this.quad = new QuadUv();
    this.state = State.for2d();
  }
  contextChange() {
    const renderer = this.renderer;
    const uniforms = { globals: renderer.globalUniforms };
    this.simpleShader = Shader.from(gl1VertexSrc, fragmentSimpleSrc, uniforms);
    this.shader = renderer.context.webGLVersion > 1 ? Shader.from(gl2VertexSrc, gl2FragmentSrc, uniforms) : Shader.from(gl1VertexSrc, gl1FragmentSrc, uniforms);
  }
  render(ts) {
    const renderer = this.renderer;
    const quad = this.quad;
    let vertices = quad.vertices;
    vertices[0] = vertices[6] = ts._width * -ts.anchor.x;
    vertices[1] = vertices[3] = ts._height * -ts.anchor.y;
    vertices[2] = vertices[4] = ts._width * (1 - ts.anchor.x);
    vertices[5] = vertices[7] = ts._height * (1 - ts.anchor.y);
    const anchorX = ts.uvRespectAnchor ? ts.anchor.x : 0;
    const anchorY = ts.uvRespectAnchor ? ts.anchor.y : 0;
    vertices = quad.uvs;
    vertices[0] = vertices[6] = -anchorX;
    vertices[1] = vertices[3] = -anchorY;
    vertices[2] = vertices[4] = 1 - anchorX;
    vertices[5] = vertices[7] = 1 - anchorY;
    quad.invalidate();
    const tex = ts._texture;
    const baseTex = tex.baseTexture;
    const premultiplied = baseTex.alphaMode > 0;
    const lt = ts.tileTransform.localTransform;
    const uv = ts.uvMatrix;
    let isSimple = baseTex.isPowerOfTwo && tex.frame.width === baseTex.width && tex.frame.height === baseTex.height;
    if (isSimple) {
      if (!baseTex._glTextures[renderer.CONTEXT_UID]) {
        if (baseTex.wrapMode === WRAP_MODES.CLAMP) {
          baseTex.wrapMode = WRAP_MODES.REPEAT;
        }
      } else {
        isSimple = baseTex.wrapMode !== WRAP_MODES.CLAMP;
      }
    }
    const shader = isSimple ? this.simpleShader : this.shader;
    const w = tex.width;
    const h = tex.height;
    const W = ts._width;
    const H = ts._height;
    tempMat.set(lt.a * w / W, lt.b * w / H, lt.c * h / W, lt.d * h / H, lt.tx / W, lt.ty / H);
    tempMat.invert();
    if (isSimple) {
      tempMat.prepend(uv.mapCoord);
    } else {
      shader.uniforms.uMapCoord = uv.mapCoord.toArray(true);
      shader.uniforms.uClampFrame = uv.uClampFrame;
      shader.uniforms.uClampOffset = uv.uClampOffset;
    }
    shader.uniforms.uTransform = tempMat.toArray(true);
    shader.uniforms.uColor = Color.shared.setValue(ts.tint).premultiply(ts.worldAlpha, premultiplied).toArray(shader.uniforms.uColor);
    shader.uniforms.translationMatrix = ts.transform.worldTransform.toArray(true);
    shader.uniforms.uSampler = tex;
    renderer.shader.bind(shader);
    renderer.geometry.bind(quad);
    this.state.blendMode = utils.correctBlendMode(ts.blendMode, premultiplied);
    renderer.state.set(this.state);
    renderer.geometry.draw(this.renderer.gl.TRIANGLES, 6, 0);
  }
}
TilingSpriteRenderer.extension = {
  name: "tilingSprite",
  type: ExtensionType.RendererPlugin
};
extensions.add(TilingSpriteRenderer);

export { TilingSpriteRenderer };
//# sourceMappingURL=TilingSpriteRenderer.mjs.map
