'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var core = require('@pixi/core');
var sprite = require('@pixi/sprite');
var text = require('@pixi/text');
var HTMLTextStyle = require('./HTMLTextStyle.js');

const _HTMLText = class extends sprite.Sprite {
  constructor(text = "", style = {}) {
    super(core.Texture.EMPTY);
    this._text = null;
    this._style = null;
    this._autoResolution = true;
    this._loading = false;
    this.localStyleID = -1;
    this.dirty = false;
    this.ownsStyle = false;
    const image = new Image();
    const texture = core.Texture.from(image, {
      scaleMode: core.settings.SCALE_MODE,
      resourceOptions: {
        autoLoad: false
      }
    });
    texture.orig = new core.Rectangle();
    texture.trim = new core.Rectangle();
    this.texture = texture;
    const nssvg = "http://www.w3.org/2000/svg";
    const nsxhtml = "http://www.w3.org/1999/xhtml";
    const svgRoot = document.createElementNS(nssvg, "svg");
    const foreignObject = document.createElementNS(nssvg, "foreignObject");
    const domElement = document.createElementNS(nsxhtml, "div");
    const styleElement = document.createElementNS(nsxhtml, "style");
    foreignObject.setAttribute("width", "10000");
    foreignObject.setAttribute("height", "10000");
    foreignObject.style.overflow = "hidden";
    svgRoot.appendChild(foreignObject);
    this.maxWidth = _HTMLText.defaultMaxWidth;
    this.maxHeight = _HTMLText.defaultMaxHeight;
    this._domElement = domElement;
    this._styleElement = styleElement;
    this._svgRoot = svgRoot;
    this._foreignObject = foreignObject;
    this._foreignObject.appendChild(styleElement);
    this._foreignObject.appendChild(domElement);
    this._image = image;
    this._loadImage = new Image();
    this._autoResolution = _HTMLText.defaultAutoResolution;
    this._resolution = _HTMLText.defaultResolution ?? core.settings.RESOLUTION;
    this.text = text;
    this.style = style;
  }
  measureText(overrides) {
    const { text, style, resolution } = Object.assign({
      text: this._text,
      style: this._style,
      resolution: this._resolution
    }, overrides);
    Object.assign(this._domElement, {
      innerHTML: text,
      style: style.toCSS(resolution)
    });
    this._styleElement.textContent = style.toGlobalCSS();
    document.body.appendChild(this._svgRoot);
    const contentBounds = this._domElement.getBoundingClientRect();
    this._svgRoot.remove();
    const contentWidth = Math.min(this.maxWidth, Math.ceil(contentBounds.width));
    const contentHeight = Math.min(this.maxHeight, Math.ceil(contentBounds.height));
    this._svgRoot.setAttribute("width", contentWidth.toString());
    this._svgRoot.setAttribute("height", contentHeight.toString());
    if (text !== this._text) {
      this._domElement.innerHTML = this._text;
    }
    if (style !== this._style) {
      Object.assign(this._domElement, { style: this._style?.toCSS(resolution) });
      this._styleElement.textContent = this._style?.toGlobalCSS();
    }
    return {
      width: contentWidth + style.padding * 2,
      height: contentHeight + style.padding * 2
    };
  }
  async updateText(respectDirty = true) {
    const { style, _image: image, _loadImage: loadImage } = this;
    if (this.localStyleID !== style.styleID) {
      this.dirty = true;
      this.localStyleID = style.styleID;
    }
    if (!this.dirty && respectDirty) {
      return;
    }
    const { width, height } = this.measureText();
    image.width = loadImage.width = Math.ceil(Math.max(1, width));
    image.height = loadImage.height = Math.ceil(Math.max(1, height));
    if (!this._loading) {
      this._loading = true;
      await new Promise((resolve) => {
        loadImage.onload = async () => {
          await style.onBeforeDraw();
          this._loading = false;
          image.src = loadImage.src;
          loadImage.onload = null;
          loadImage.src = "";
          this.updateTexture();
          resolve();
        };
        const svgURL = new XMLSerializer().serializeToString(this._svgRoot);
        loadImage.src = `data:image/svg+xml;charset=utf8,${encodeURIComponent(svgURL)}`;
      });
    }
  }
  get source() {
    return this._image;
  }
  updateTexture() {
    const { style, texture, _image: image, resolution } = this;
    const { padding } = style;
    const { baseTexture } = texture;
    texture.trim.width = texture._frame.width = image.width / resolution;
    texture.trim.height = texture._frame.height = image.height / resolution;
    texture.trim.x = -padding;
    texture.trim.y = -padding;
    texture.orig.width = texture._frame.width - padding * 2;
    texture.orig.height = texture._frame.height - padding * 2;
    this._onTextureUpdate();
    baseTexture.setRealSize(image.width, image.height, resolution);
    this.dirty = false;
  }
  _render(renderer) {
    if (this._autoResolution && this._resolution !== renderer.resolution) {
      this._resolution = renderer.resolution;
      this.dirty = true;
    }
    this.updateText(true);
    super._render(renderer);
  }
  _renderCanvas(renderer) {
    if (this._autoResolution && this._resolution !== renderer.resolution) {
      this._resolution = renderer.resolution;
      this.dirty = true;
    }
    this.updateText(true);
    super._renderCanvas(renderer);
  }
  getLocalBounds(rect) {
    this.updateText(true);
    return super.getLocalBounds(rect);
  }
  _calculateBounds() {
    this.updateText(true);
    this.calculateVertices();
    this._bounds.addQuad(this.vertexData);
  }
  _onStyleChange() {
    this.dirty = true;
  }
  destroy(options) {
    if (typeof options === "boolean") {
      options = { children: options };
    }
    options = Object.assign({}, _HTMLText.defaultDestroyOptions, options);
    super.destroy(options);
    const forceClear = null;
    if (this.ownsStyle) {
      this._style?.cleanFonts();
    }
    this._style = forceClear;
    this._svgRoot?.remove();
    this._svgRoot = forceClear;
    this._domElement?.remove();
    this._domElement = forceClear;
    this._foreignObject?.remove();
    this._foreignObject = forceClear;
    this._styleElement?.remove();
    this._styleElement = forceClear;
    this._loadImage.src = "";
    this._loadImage.onload = null;
    this._loadImage = forceClear;
    this._image.src = "";
    this._image = forceClear;
  }
  get width() {
    this.updateText(true);
    return Math.abs(this.scale.x) * this._image.width / this.resolution;
  }
  set width(value) {
    this.updateText(true);
    const s = core.utils.sign(this.scale.x) || 1;
    this.scale.x = s * value / this._image.width / this.resolution;
    this._width = value;
  }
  get height() {
    this.updateText(true);
    return Math.abs(this.scale.y) * this._image.height / this.resolution;
  }
  set height(value) {
    this.updateText(true);
    const s = core.utils.sign(this.scale.y) || 1;
    this.scale.y = s * value / this._image.height / this.resolution;
    this._height = value;
  }
  get style() {
    return this._style;
  }
  set style(style) {
    if (this._style === style) {
      return;
    }
    style = style || {};
    if (style instanceof HTMLTextStyle.HTMLTextStyle) {
      this.ownsStyle = false;
      this._style = style;
    } else if (style instanceof text.TextStyle) {
      console.warn("[HTMLText] Cloning TextStyle, if this is not what you want, use HTMLTextStyle");
      this.ownsStyle = true;
      this._style = HTMLTextStyle.HTMLTextStyle.from(style);
    } else {
      this.ownsStyle = true;
      this._style = new HTMLTextStyle.HTMLTextStyle(style);
    }
    this.localStyleID = -1;
    this.dirty = true;
  }
  get text() {
    return this._text;
  }
  set text(text) {
    text = String(text === "" || text === null || text === void 0 ? " " : text);
    text = this.sanitiseText(text);
    if (this._text === text) {
      return;
    }
    this._text = text;
    this.dirty = true;
  }
  get resolution() {
    return this._resolution;
  }
  set resolution(value) {
    this._autoResolution = false;
    if (this._resolution === value) {
      return;
    }
    this._resolution = value;
    this.dirty = true;
  }
  sanitiseText(text) {
    return text.replace(/<br>/gi, "<br/>").replace(/<hr>/gi, "<hr/>").replace(/&nbsp;/gi, "&#160;");
  }
};
let HTMLText = _HTMLText;
HTMLText.defaultDestroyOptions = {
  texture: true,
  children: false,
  baseTexture: true
};
HTMLText.defaultMaxWidth = 2024;
HTMLText.defaultMaxHeight = 2024;
HTMLText.defaultAutoResolution = true;

exports.HTMLText = HTMLText;
//# sourceMappingURL=HTMLText.js.map
