import { settings, utils } from '@pixi/core';
import { TextStyle } from '@pixi/text';

const _HTMLTextStyle = class extends TextStyle {
  constructor() {
    super(...arguments);
    this._fonts = [];
    this._overrides = [];
    this._stylesheet = "";
    this.fontsDirty = false;
  }
  static from(originalStyle) {
    return new _HTMLTextStyle(Object.keys(_HTMLTextStyle.defaultOptions).reduce((obj, prop) => ({ ...obj, [prop]: originalStyle[prop] }), {}));
  }
  cleanFonts() {
    if (this._fonts.length > 0) {
      this._fonts.forEach((font) => {
        URL.revokeObjectURL(font.src);
        font.refs--;
        if (font.refs === 0) {
          if (font.fontFace) {
            document.fonts.delete(font.fontFace);
          }
          delete _HTMLTextStyle.availableFonts[font.originalUrl];
        }
      });
      this.fontFamily = "Arial";
      this._fonts.length = 0;
      this.styleID++;
      this.fontsDirty = true;
    }
  }
  loadFont(url, options = {}) {
    const { availableFonts } = _HTMLTextStyle;
    if (availableFonts[url]) {
      const font = availableFonts[url];
      this._fonts.push(font);
      font.refs++;
      this.styleID++;
      this.fontsDirty = true;
      return Promise.resolve();
    }
    return settings.ADAPTER.fetch(url).then((response) => response.blob()).then(async (blob) => new Promise((resolve, reject) => {
      const src = URL.createObjectURL(blob);
      const reader = new FileReader();
      reader.onload = () => resolve([src, reader.result]);
      reader.onerror = reject;
      reader.readAsDataURL(blob);
    })).then(async ([src, dataSrc]) => {
      const font = Object.assign({
        family: utils.path.basename(url, utils.path.extname(url)),
        weight: "normal",
        style: "normal",
        src,
        dataSrc,
        refs: 1,
        originalUrl: url,
        fontFace: null
      }, options);
      availableFonts[url] = font;
      this._fonts.push(font);
      this.styleID++;
      const fontFace = new FontFace(font.family, `url(${font.src})`, {
        weight: font.weight,
        style: font.style
      });
      font.fontFace = fontFace;
      await fontFace.load();
      document.fonts.add(fontFace);
      await document.fonts.ready;
      this.styleID++;
      this.fontsDirty = true;
    });
  }
  addOverride(...value) {
    const toAdd = value.filter((v) => !this._overrides.includes(v));
    if (toAdd.length > 0) {
      this._overrides.push(...toAdd);
      this.styleID++;
    }
  }
  removeOverride(...value) {
    const toRemove = value.filter((v) => this._overrides.includes(v));
    if (toRemove.length > 0) {
      this._overrides = this._overrides.filter((v) => !toRemove.includes(v));
      this.styleID++;
    }
  }
  toCSS(scale) {
    return [
      `transform: scale(${scale})`,
      `transform-origin: top left`,
      "display: inline-block",
      `color: ${this.normalizeColor(this.fill)}`,
      `font-size: ${this.fontSize}px`,
      `font-family: ${this.fontFamily}`,
      `font-weight: ${this.fontWeight}`,
      `font-style: ${this.fontStyle}`,
      `font-variant: ${this.fontVariant}`,
      `letter-spacing: ${this.letterSpacing}px`,
      `text-align: ${this.align}`,
      `padding: ${this.padding}px`,
      `white-space: ${this.whiteSpace}`,
      ...this.lineHeight ? [`line-height: ${this.lineHeight}px`] : [],
      ...this.wordWrap ? [
        `word-wrap: ${this.breakWords ? "break-all" : "break-word"}`,
        `max-width: ${this.wordWrapWidth}px`
      ] : [],
      ...this.strokeThickness ? [
        `-webkit-text-stroke-width: ${this.strokeThickness}px`,
        `-webkit-text-stroke-color: ${this.normalizeColor(this.stroke)}`,
        `text-stroke-width: ${this.strokeThickness}px`,
        `text-stroke-color: ${this.normalizeColor(this.stroke)}`,
        "paint-order: stroke"
      ] : [],
      ...this.dropShadow ? [this.dropShadowToCSS()] : [],
      ...this._overrides
    ].join(";");
  }
  toGlobalCSS() {
    return this._fonts.reduce((result, font) => `${result}
            @font-face {
                font-family: "${font.family}";
                src: url('${font.dataSrc}');
                font-weight: ${font.weight};
                font-style: ${font.style}; 
            }`, this._stylesheet);
  }
  get stylesheet() {
    return this._stylesheet;
  }
  set stylesheet(value) {
    if (this._stylesheet !== value) {
      this._stylesheet = value;
      this.styleID++;
    }
  }
  normalizeColor(color) {
    if (Array.isArray(color)) {
      color = utils.rgb2hex(color);
    }
    if (typeof color === "number") {
      return utils.hex2string(color);
    }
    return color;
  }
  dropShadowToCSS() {
    let color = this.normalizeColor(this.dropShadowColor);
    const alpha = this.dropShadowAlpha;
    const x = Math.round(Math.cos(this.dropShadowAngle) * this.dropShadowDistance);
    const y = Math.round(Math.sin(this.dropShadowAngle) * this.dropShadowDistance);
    if (color.startsWith("#") && alpha < 1) {
      color += (alpha * 255 | 0).toString(16).padStart(2, "0");
    }
    const position = `${x}px ${y}px`;
    if (this.dropShadowBlur > 0) {
      return `text-shadow: ${position} ${this.dropShadowBlur}px ${color}`;
    }
    return `text-shadow: ${position} ${color}`;
  }
  reset() {
    Object.assign(this, _HTMLTextStyle.defaultOptions);
  }
  onBeforeDraw() {
    const { fontsDirty: prevFontsDirty } = this;
    this.fontsDirty = false;
    if (this.isSafari && this._fonts.length > 0 && prevFontsDirty) {
      return new Promise((resolve) => setTimeout(resolve, 100));
    }
    return Promise.resolve();
  }
  get isSafari() {
    const { userAgent } = settings.ADAPTER.getNavigator();
    return /^((?!chrome|android).)*safari/i.test(userAgent);
  }
  set fillGradientStops(_value) {
    console.warn("[HTMLTextStyle] fillGradientStops is not supported by HTMLText");
  }
  get fillGradientStops() {
    return super.fillGradientStops;
  }
  set fillGradientType(_value) {
    console.warn("[HTMLTextStyle] fillGradientType is not supported by HTMLText");
  }
  get fillGradientType() {
    return super.fillGradientType;
  }
  set miterLimit(_value) {
    console.warn("[HTMLTextStyle] miterLimit is not supported by HTMLText");
  }
  get miterLimit() {
    return super.miterLimit;
  }
  set trim(_value) {
    console.warn("[HTMLTextStyle] trim is not supported by HTMLText");
  }
  get trim() {
    return super.trim;
  }
  set textBaseline(_value) {
    console.warn("[HTMLTextStyle] textBaseline is not supported by HTMLText");
  }
  get textBaseline() {
    return super.textBaseline;
  }
  set leading(_value) {
    console.warn("[HTMLTextStyle] leading is not supported by HTMLText");
  }
  get leading() {
    return super.leading;
  }
  set lineJoin(_value) {
    console.warn("[HTMLTextStyle] lineJoin is not supported by HTMLText");
  }
  get lineJoin() {
    return super.lineJoin;
  }
};
let HTMLTextStyle = _HTMLTextStyle;
HTMLTextStyle.availableFonts = {};
HTMLTextStyle.defaultOptions = {
  align: "left",
  breakWords: false,
  dropShadow: false,
  dropShadowAlpha: 1,
  dropShadowAngle: Math.PI / 6,
  dropShadowBlur: 0,
  dropShadowColor: "black",
  dropShadowDistance: 5,
  fill: "black",
  fontFamily: "Arial",
  fontSize: 26,
  fontStyle: "normal",
  fontVariant: "normal",
  fontWeight: "normal",
  letterSpacing: 0,
  lineHeight: 0,
  padding: 0,
  stroke: "black",
  strokeThickness: 0,
  whiteSpace: "normal",
  wordWrap: false,
  wordWrapWidth: 100
};

export { HTMLTextStyle };
//# sourceMappingURL=HTMLTextStyle.mjs.map
