import { utils, ALPHA_MODES, MIPMAP_MODES, Rectangle, Texture, settings, BaseTexture } from '@pixi/core';
import { TextStyle, TextMetrics } from '@pixi/text';
import { BitmapFontData } from './BitmapFontData.mjs';
import { autoDetectFormat } from './formats/index.mjs';
import './utils/index.mjs';
import { resolveCharacters } from './utils/resolveCharacters.mjs';
import { drawGlyph } from './utils/drawGlyph.mjs';
import { extractCharCode } from './utils/extractCharCode.mjs';

const _BitmapFont = class {
  constructor(data, textures, ownsTextures) {
    const [info] = data.info;
    const [common] = data.common;
    const [page] = data.page;
    const [distanceField] = data.distanceField;
    const res = utils.getResolutionOfUrl(page.file);
    const pageTextures = {};
    this._ownsTextures = ownsTextures;
    this.font = info.face;
    this.size = info.size;
    this.lineHeight = common.lineHeight / res;
    this.chars = {};
    this.pageTextures = pageTextures;
    for (let i = 0; i < data.page.length; i++) {
      const { id, file } = data.page[i];
      pageTextures[id] = textures instanceof Array ? textures[i] : textures[file];
      if (distanceField?.fieldType && distanceField.fieldType !== "none") {
        pageTextures[id].baseTexture.alphaMode = ALPHA_MODES.NO_PREMULTIPLIED_ALPHA;
        pageTextures[id].baseTexture.mipmap = MIPMAP_MODES.OFF;
      }
    }
    for (let i = 0; i < data.char.length; i++) {
      const { id, page: page2 } = data.char[i];
      let { x, y, width, height, xoffset, yoffset, xadvance } = data.char[i];
      x /= res;
      y /= res;
      width /= res;
      height /= res;
      xoffset /= res;
      yoffset /= res;
      xadvance /= res;
      const rect = new Rectangle(x + pageTextures[page2].frame.x / res, y + pageTextures[page2].frame.y / res, width, height);
      this.chars[id] = {
        xOffset: xoffset,
        yOffset: yoffset,
        xAdvance: xadvance,
        kerning: {},
        texture: new Texture(pageTextures[page2].baseTexture, rect),
        page: page2
      };
    }
    for (let i = 0; i < data.kerning.length; i++) {
      let { first, second, amount } = data.kerning[i];
      first /= res;
      second /= res;
      amount /= res;
      if (this.chars[second]) {
        this.chars[second].kerning[first] = amount;
      }
    }
    this.distanceFieldRange = distanceField?.distanceRange;
    this.distanceFieldType = distanceField?.fieldType?.toLowerCase() ?? "none";
  }
  destroy() {
    for (const id in this.chars) {
      this.chars[id].texture.destroy();
      this.chars[id].texture = null;
    }
    for (const id in this.pageTextures) {
      if (this._ownsTextures) {
        this.pageTextures[id].destroy(true);
      }
      this.pageTextures[id] = null;
    }
    this.chars = null;
    this.pageTextures = null;
  }
  static install(data, textures, ownsTextures) {
    let fontData;
    if (data instanceof BitmapFontData) {
      fontData = data;
    } else {
      const format = autoDetectFormat(data);
      if (!format) {
        throw new Error("Unrecognized data format for font.");
      }
      fontData = format.parse(data);
    }
    if (textures instanceof Texture) {
      textures = [textures];
    }
    const font = new _BitmapFont(fontData, textures, ownsTextures);
    _BitmapFont.available[font.font] = font;
    return font;
  }
  static uninstall(name) {
    const font = _BitmapFont.available[name];
    if (!font) {
      throw new Error(`No font found named '${name}'`);
    }
    font.destroy();
    delete _BitmapFont.available[name];
  }
  static from(name, textStyle, options) {
    if (!name) {
      throw new Error("[BitmapFont] Property `name` is required.");
    }
    const {
      chars,
      padding,
      resolution,
      textureWidth,
      textureHeight,
      ...baseOptions
    } = Object.assign({}, _BitmapFont.defaultOptions, options);
    const charsList = resolveCharacters(chars);
    const style = textStyle instanceof TextStyle ? textStyle : new TextStyle(textStyle);
    const lineWidth = textureWidth;
    const fontData = new BitmapFontData();
    fontData.info[0] = {
      face: style.fontFamily,
      size: style.fontSize
    };
    fontData.common[0] = {
      lineHeight: style.fontSize
    };
    let positionX = 0;
    let positionY = 0;
    let canvas;
    let context;
    let baseTexture;
    let maxCharHeight = 0;
    const baseTextures = [];
    const textures = [];
    for (let i = 0; i < charsList.length; i++) {
      if (!canvas) {
        canvas = settings.ADAPTER.createCanvas();
        canvas.width = textureWidth;
        canvas.height = textureHeight;
        context = canvas.getContext("2d");
        baseTexture = new BaseTexture(canvas, { resolution, ...baseOptions });
        baseTextures.push(baseTexture);
        textures.push(new Texture(baseTexture));
        fontData.page.push({
          id: textures.length - 1,
          file: ""
        });
      }
      const character = charsList[i];
      const metrics = TextMetrics.measureText(character, style, false, canvas);
      const width = metrics.width;
      const height = Math.ceil(metrics.height);
      const textureGlyphWidth = Math.ceil((style.fontStyle === "italic" ? 2 : 1) * width);
      if (positionY >= textureHeight - height * resolution) {
        if (positionY === 0) {
          throw new Error(`[BitmapFont] textureHeight ${textureHeight}px is too small (fontFamily: '${style.fontFamily}', fontSize: ${style.fontSize}px, char: '${character}')`);
        }
        --i;
        canvas = null;
        context = null;
        baseTexture = null;
        positionY = 0;
        positionX = 0;
        maxCharHeight = 0;
        continue;
      }
      maxCharHeight = Math.max(height + metrics.fontProperties.descent, maxCharHeight);
      if (textureGlyphWidth * resolution + positionX >= lineWidth) {
        if (positionX === 0) {
          throw new Error(`[BitmapFont] textureWidth ${textureWidth}px is too small (fontFamily: '${style.fontFamily}', fontSize: ${style.fontSize}px, char: '${character}')`);
        }
        --i;
        positionY += maxCharHeight * resolution;
        positionY = Math.ceil(positionY);
        positionX = 0;
        maxCharHeight = 0;
        continue;
      }
      drawGlyph(canvas, context, metrics, positionX, positionY, resolution, style);
      const id = extractCharCode(metrics.text);
      fontData.char.push({
        id,
        page: textures.length - 1,
        x: positionX / resolution,
        y: positionY / resolution,
        width: textureGlyphWidth,
        height,
        xoffset: 0,
        yoffset: 0,
        xadvance: width - (style.dropShadow ? style.dropShadowDistance : 0) - (style.stroke ? style.strokeThickness : 0)
      });
      positionX += (textureGlyphWidth + 2 * padding) * resolution;
      positionX = Math.ceil(positionX);
    }
    for (let i = 0, len = charsList.length; i < len; i++) {
      const first = charsList[i];
      for (let j = 0; j < len; j++) {
        const second = charsList[j];
        const c1 = context.measureText(first).width;
        const c2 = context.measureText(second).width;
        const total = context.measureText(first + second).width;
        const amount = total - (c1 + c2);
        if (amount) {
          fontData.kerning.push({
            first: extractCharCode(first),
            second: extractCharCode(second),
            amount
          });
        }
      }
    }
    const font = new _BitmapFont(fontData, textures, true);
    if (_BitmapFont.available[name] !== void 0) {
      _BitmapFont.uninstall(name);
    }
    _BitmapFont.available[name] = font;
    return font;
  }
};
let BitmapFont = _BitmapFont;
BitmapFont.ALPHA = [["a", "z"], ["A", "Z"], " "];
BitmapFont.NUMERIC = [["0", "9"]];
BitmapFont.ALPHANUMERIC = [["a", "z"], ["A", "Z"], ["0", "9"], " "];
BitmapFont.ASCII = [[" ", "~"]];
BitmapFont.defaultOptions = {
  resolution: 1,
  textureWidth: 512,
  textureHeight: 512,
  padding: 4,
  chars: _BitmapFont.ALPHANUMERIC
};
BitmapFont.available = {};

export { BitmapFont };
//# sourceMappingURL=BitmapFont.mjs.map
