'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var core = require('@pixi/core');

const TEMP_RECT = new core.Rectangle();
const BYTES_PER_PIXEL = 4;
const _Extract = class {
  constructor(renderer) {
    this.renderer = renderer;
  }
  async image(target, format, quality) {
    const image = new Image();
    image.src = await this.base64(target, format, quality);
    return image;
  }
  async base64(target, format, quality) {
    const canvas = this.canvas(target);
    if (canvas.toBlob !== void 0) {
      return new Promise((resolve, reject) => {
        canvas.toBlob((blob) => {
          if (!blob) {
            reject(new Error("ICanvas.toBlob failed!"));
            return;
          }
          const reader = new FileReader();
          reader.onload = () => resolve(reader.result);
          reader.onerror = reject;
          reader.readAsDataURL(blob);
        }, format, quality);
      });
    }
    if (canvas.toDataURL !== void 0) {
      return canvas.toDataURL(format, quality);
    }
    if (canvas.convertToBlob !== void 0) {
      const blob = await canvas.convertToBlob({ type: format, quality });
      return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
      });
    }
    throw new Error("Extract.base64() requires ICanvas.toDataURL, ICanvas.toBlob, or ICanvas.convertToBlob to be implemented");
  }
  canvas(target, frame) {
    const { pixels, width, height, flipY } = this._rawPixels(target, frame);
    if (flipY) {
      _Extract._flipY(pixels, width, height);
    }
    _Extract._unpremultiplyAlpha(pixels);
    const canvasBuffer = new core.utils.CanvasRenderTarget(width, height, 1);
    const imageData = new ImageData(new Uint8ClampedArray(pixels.buffer), width, height);
    canvasBuffer.context.putImageData(imageData, 0, 0);
    return canvasBuffer.canvas;
  }
  pixels(target, frame) {
    const { pixels, width, height, flipY } = this._rawPixels(target, frame);
    if (flipY) {
      _Extract._flipY(pixels, width, height);
    }
    _Extract._unpremultiplyAlpha(pixels);
    return pixels;
  }
  _rawPixels(target, frame) {
    const renderer = this.renderer;
    if (!renderer) {
      throw new Error("The Extract has already been destroyed");
    }
    let resolution;
    let flipY = false;
    let renderTexture;
    let generated = false;
    if (target) {
      if (target instanceof core.RenderTexture) {
        renderTexture = target;
      } else {
        renderTexture = renderer.generateTexture(target, {
          resolution: renderer.resolution,
          multisample: renderer.multisample
        });
        generated = true;
      }
    }
    if (renderTexture) {
      resolution = renderTexture.baseTexture.resolution;
      frame = frame ?? renderTexture.frame;
      flipY = false;
      if (!generated) {
        renderer.renderTexture.bind(renderTexture);
        const fbo = renderTexture.framebuffer.glFramebuffers[renderer.CONTEXT_UID];
        if (fbo.blitFramebuffer) {
          renderer.framebuffer.bind(fbo.blitFramebuffer);
        }
      }
    } else {
      resolution = renderer.resolution;
      if (!frame) {
        frame = TEMP_RECT;
        frame.width = renderer.width / resolution;
        frame.height = renderer.height / resolution;
      }
      flipY = true;
      renderer.renderTexture.bind();
    }
    const width = Math.round(frame.width * resolution);
    const height = Math.round(frame.height * resolution);
    const pixels = new Uint8Array(BYTES_PER_PIXEL * width * height);
    const gl = renderer.gl;
    gl.readPixels(Math.round(frame.x * resolution), Math.round(frame.y * resolution), width, height, gl.RGBA, gl.UNSIGNED_BYTE, pixels);
    if (generated) {
      renderTexture?.destroy(true);
    }
    return { pixels, width, height, flipY };
  }
  destroy() {
    this.renderer = null;
  }
  static _flipY(pixels, width, height) {
    const w = width << 2;
    const h = height >> 1;
    const temp = new Uint8Array(w);
    for (let y = 0; y < h; y++) {
      const t = y * w;
      const b = (height - y - 1) * w;
      temp.set(pixels.subarray(t, t + w));
      pixels.copyWithin(t, b, b + w);
      pixels.set(temp, b);
    }
  }
  static _unpremultiplyAlpha(pixels) {
    if (pixels instanceof Uint8ClampedArray) {
      pixels = new Uint8Array(pixels.buffer);
    }
    const n = pixels.length;
    for (let i = 0; i < n; i += 4) {
      const alpha = pixels[i + 3];
      if (alpha !== 0) {
        const a = 255.001 / alpha;
        pixels[i] = pixels[i] * a + 0.5;
        pixels[i + 1] = pixels[i + 1] * a + 0.5;
        pixels[i + 2] = pixels[i + 2] * a + 0.5;
      }
    }
  }
};
let Extract = _Extract;
Extract.extension = {
  name: "extract",
  type: core.ExtensionType.RendererSystem
};
core.extensions.add(Extract);

exports.Extract = Extract;
//# sourceMappingURL=Extract.js.map
