import { ObjectRenderer, Matrix, TYPES, Shader, State, utils, Color, ExtensionType, extensions } from '@pixi/core';
import { ParticleBuffer } from './ParticleBuffer.mjs';
import fragment from './particles.mjs';
import vertex from './particles2.mjs';

class ParticleRenderer extends ObjectRenderer {
  constructor(renderer) {
    super(renderer);
    this.shader = null;
    this.properties = null;
    this.tempMatrix = new Matrix();
    this.properties = [
      {
        attributeName: "aVertexPosition",
        size: 2,
        uploadFunction: this.uploadVertices,
        offset: 0
      },
      {
        attributeName: "aPositionCoord",
        size: 2,
        uploadFunction: this.uploadPosition,
        offset: 0
      },
      {
        attributeName: "aRotation",
        size: 1,
        uploadFunction: this.uploadRotation,
        offset: 0
      },
      {
        attributeName: "aTextureCoord",
        size: 2,
        uploadFunction: this.uploadUvs,
        offset: 0
      },
      {
        attributeName: "aColor",
        size: 1,
        type: TYPES.UNSIGNED_BYTE,
        uploadFunction: this.uploadTint,
        offset: 0
      }
    ];
    this.shader = Shader.from(vertex, fragment, {});
    this.state = State.for2d();
  }
  render(container) {
    const children = container.children;
    const maxSize = container._maxSize;
    const batchSize = container._batchSize;
    const renderer = this.renderer;
    let totalChildren = children.length;
    if (totalChildren === 0) {
      return;
    } else if (totalChildren > maxSize && !container.autoResize) {
      totalChildren = maxSize;
    }
    let buffers = container._buffers;
    if (!buffers) {
      buffers = container._buffers = this.generateBuffers(container);
    }
    const baseTexture = children[0]._texture.baseTexture;
    const premultiplied = baseTexture.alphaMode > 0;
    this.state.blendMode = utils.correctBlendMode(container.blendMode, premultiplied);
    renderer.state.set(this.state);
    const gl = renderer.gl;
    const m = container.worldTransform.copyTo(this.tempMatrix);
    m.prepend(renderer.globalUniforms.uniforms.projectionMatrix);
    this.shader.uniforms.translationMatrix = m.toArray(true);
    this.shader.uniforms.uColor = Color.shared.setValue(container.tintRgb).premultiply(container.worldAlpha, premultiplied).toArray(this.shader.uniforms.uColor);
    this.shader.uniforms.uSampler = baseTexture;
    this.renderer.shader.bind(this.shader);
    let updateStatic = false;
    for (let i = 0, j = 0; i < totalChildren; i += batchSize, j += 1) {
      let amount = totalChildren - i;
      if (amount > batchSize) {
        amount = batchSize;
      }
      if (j >= buffers.length) {
        buffers.push(this._generateOneMoreBuffer(container));
      }
      const buffer = buffers[j];
      buffer.uploadDynamic(children, i, amount);
      const bid = container._bufferUpdateIDs[j] || 0;
      updateStatic = updateStatic || buffer._updateID < bid;
      if (updateStatic) {
        buffer._updateID = container._updateID;
        buffer.uploadStatic(children, i, amount);
      }
      renderer.geometry.bind(buffer.geometry);
      gl.drawElements(gl.TRIANGLES, amount * 6, gl.UNSIGNED_SHORT, 0);
    }
  }
  generateBuffers(container) {
    const buffers = [];
    const size = container._maxSize;
    const batchSize = container._batchSize;
    const dynamicPropertyFlags = container._properties;
    for (let i = 0; i < size; i += batchSize) {
      buffers.push(new ParticleBuffer(this.properties, dynamicPropertyFlags, batchSize));
    }
    return buffers;
  }
  _generateOneMoreBuffer(container) {
    const batchSize = container._batchSize;
    const dynamicPropertyFlags = container._properties;
    return new ParticleBuffer(this.properties, dynamicPropertyFlags, batchSize);
  }
  uploadVertices(children, startIndex, amount, array, stride, offset) {
    let w0 = 0;
    let w1 = 0;
    let h0 = 0;
    let h1 = 0;
    for (let i = 0; i < amount; ++i) {
      const sprite = children[startIndex + i];
      const texture = sprite._texture;
      const sx = sprite.scale.x;
      const sy = sprite.scale.y;
      const trim = texture.trim;
      const orig = texture.orig;
      if (trim) {
        w1 = trim.x - sprite.anchor.x * orig.width;
        w0 = w1 + trim.width;
        h1 = trim.y - sprite.anchor.y * orig.height;
        h0 = h1 + trim.height;
      } else {
        w0 = orig.width * (1 - sprite.anchor.x);
        w1 = orig.width * -sprite.anchor.x;
        h0 = orig.height * (1 - sprite.anchor.y);
        h1 = orig.height * -sprite.anchor.y;
      }
      array[offset] = w1 * sx;
      array[offset + 1] = h1 * sy;
      array[offset + stride] = w0 * sx;
      array[offset + stride + 1] = h1 * sy;
      array[offset + stride * 2] = w0 * sx;
      array[offset + stride * 2 + 1] = h0 * sy;
      array[offset + stride * 3] = w1 * sx;
      array[offset + stride * 3 + 1] = h0 * sy;
      offset += stride * 4;
    }
  }
  uploadPosition(children, startIndex, amount, array, stride, offset) {
    for (let i = 0; i < amount; i++) {
      const spritePosition = children[startIndex + i].position;
      array[offset] = spritePosition.x;
      array[offset + 1] = spritePosition.y;
      array[offset + stride] = spritePosition.x;
      array[offset + stride + 1] = spritePosition.y;
      array[offset + stride * 2] = spritePosition.x;
      array[offset + stride * 2 + 1] = spritePosition.y;
      array[offset + stride * 3] = spritePosition.x;
      array[offset + stride * 3 + 1] = spritePosition.y;
      offset += stride * 4;
    }
  }
  uploadRotation(children, startIndex, amount, array, stride, offset) {
    for (let i = 0; i < amount; i++) {
      const spriteRotation = children[startIndex + i].rotation;
      array[offset] = spriteRotation;
      array[offset + stride] = spriteRotation;
      array[offset + stride * 2] = spriteRotation;
      array[offset + stride * 3] = spriteRotation;
      offset += stride * 4;
    }
  }
  uploadUvs(children, startIndex, amount, array, stride, offset) {
    for (let i = 0; i < amount; ++i) {
      const textureUvs = children[startIndex + i]._texture._uvs;
      if (textureUvs) {
        array[offset] = textureUvs.x0;
        array[offset + 1] = textureUvs.y0;
        array[offset + stride] = textureUvs.x1;
        array[offset + stride + 1] = textureUvs.y1;
        array[offset + stride * 2] = textureUvs.x2;
        array[offset + stride * 2 + 1] = textureUvs.y2;
        array[offset + stride * 3] = textureUvs.x3;
        array[offset + stride * 3 + 1] = textureUvs.y3;
        offset += stride * 4;
      } else {
        array[offset] = 0;
        array[offset + 1] = 0;
        array[offset + stride] = 0;
        array[offset + stride + 1] = 0;
        array[offset + stride * 2] = 0;
        array[offset + stride * 2 + 1] = 0;
        array[offset + stride * 3] = 0;
        array[offset + stride * 3 + 1] = 0;
        offset += stride * 4;
      }
    }
  }
  uploadTint(children, startIndex, amount, array, stride, offset) {
    for (let i = 0; i < amount; ++i) {
      const sprite = children[startIndex + i];
      const result = Color.shared.setValue(sprite._tintRGB).toPremultiplied(sprite.alpha, sprite.texture.baseTexture.alphaMode > 0);
      array[offset] = result;
      array[offset + stride] = result;
      array[offset + stride * 2] = result;
      array[offset + stride * 3] = result;
      offset += stride * 4;
    }
  }
  destroy() {
    super.destroy();
    if (this.shader) {
      this.shader.destroy();
      this.shader = null;
    }
    this.tempMatrix = null;
  }
}
ParticleRenderer.extension = {
  name: "particle",
  type: ExtensionType.RendererPlugin
};
extensions.add(ParticleRenderer);

export { ParticleRenderer };
//# sourceMappingURL=ParticleRenderer.mjs.map
