import { settings } from '@pixi/settings';

class CanvasRenderTarget {
  constructor(width, height, resolution) {
    this._canvas = settings.ADAPTER.createCanvas();
    this._context = this._canvas.getContext("2d");
    this.resolution = resolution || settings.RESOLUTION;
    this.resize(width, height);
  }
  clear() {
    this._checkDestroyed();
    this._context.setTransform(1, 0, 0, 1, 0, 0);
    this._context.clearRect(0, 0, this._canvas.width, this._canvas.height);
  }
  resize(desiredWidth, desiredHeight) {
    this._checkDestroyed();
    this._canvas.width = Math.round(desiredWidth * this.resolution);
    this._canvas.height = Math.round(desiredHeight * this.resolution);
  }
  destroy() {
    this._context = null;
    this._canvas = null;
  }
  get width() {
    this._checkDestroyed();
    return this._canvas.width;
  }
  set width(val) {
    this._checkDestroyed();
    this._canvas.width = Math.round(val);
  }
  get height() {
    this._checkDestroyed();
    return this._canvas.height;
  }
  set height(val) {
    this._checkDestroyed();
    this._canvas.height = Math.round(val);
  }
  get canvas() {
    this._checkDestroyed();
    return this._canvas;
  }
  get context() {
    this._checkDestroyed();
    return this._context;
  }
  _checkDestroyed() {
    if (this._canvas === null) {
      throw new TypeError("The CanvasRenderTarget has already been destroyed");
    }
  }
}

export { CanvasRenderTarget };
//# sourceMappingURL=CanvasRenderTarget.mjs.map
